import ij.IJ;
import ij.ImagePlus;
import ij.gui.GenericDialog;
import ij.macro.Interpreter;
import ij.macro.Functions;
import ij.macro.MacroExtension;
import ij.macro.ExtensionDescriptor;
import ij.plugin.PlugIn;
import ij.process.ColorProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import ij.io.OpenDialog;
import ij.text.TextPanel;
import ij.text.TextWindow;
import ij.WindowManager;
import java.awt.Choice;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Frame;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.IndexColorModel;
import java.io.BufferedReader;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Vector;
import java.text.NumberFormat;
import jama.Matrix;
import mathcs.fft.DoubleFFT_2D;


/*
This is the main code for doing the FTTC calculation as an ImageJ plugin. 
The same package contains a plotting class as well. You will need it as well as the jama and jtransform to compile the binary. 
For more detail please see:
https://sites.google.com/site/qingzongtseng/tfm
or contact me. 
By TSENG Qingzong (qztseng /at/ gmail.com)
*/


public class FTTC_main implements PlugIn, MacroExtension
{
	private static boolean			plugin							= false;
//	private static double			pixel_size						= 8.0E-6 / 60;		//	For the Andor 885
	private static double			pixel_size						= 6.5E-6 / 60;		//	For the Hamamatsu ORCA-Flash4.0 V2+
	private static double			distance_between_points_in_pixel;					// distance between each point of displacement (in pixel)
	private static double			distance_between_points_in_m;						// distance between each point of displacement (in m)
	private static double			window_area;										// (vs3 x pixel2micron)^2
	private static double			mu								= 0.5;				// the poission ratio of gel
	private static double			E								= 17000;			// Young's modulus of gel, in Pascal
	private static double			lambda							= 1e-10;			// modified for test 11-02-16
	private static boolean			plotDraw;
	private static boolean			plotScalebar;
	private static boolean			plotVector;
	private static boolean			plotMagnitude;
	private static boolean			plotMagnitudeVector;
	private static boolean			plotMagnitudeX;
	private static boolean			plotMagnitudeY;
	private static int				pictureWidth, pictureHeight;
	private static int				insetWidth, insetHeight;
	private static String			lut				= "S_Pet";
	private static String[]			pathname;
	private static String			dir, file;
	private static IndexColorModel	cm;
	private static int[]			dim = null ;										// the real distance between each point (in micron)

	private static int				EIndex			= 2;								// index of initial selected Young's modulus value in dropdown menu
	private static String[]			EText			= { "460      Pa", "1500    Pa", "17000  Pa           "};
	private static double[]			EValues			= {  460         ,  1500       , 17000  };
	
	private static int				lambdaIndex		= 1;								// index of initial selected lambda value in dropdown menu
	private static String[]			lambdaText		= { "0", "1e-10                   ", "0.01"};
	private static double[]			lambdaValues	= {  0 ,  1e-10 ,  0.01 };

	private static int				unitIndex		= 0;								// index of initial selected unit value in dropdown menu
	private static String[]			unitText		= { "pN", "pPa", "pPa and pN        "};

	public void run(String arg)
	{
		try 
		{
			pathname = FTTC_plot.getFilePath("Select the PIV displacement file");
		}
		catch (Exception e)
		{
			IJ.error(e.getMessage());
			return;
		}
		dir		= pathname[0];
		file	= pathname[1];

		generateFTTCdata();

		if (!IJ.macroRunning() || Interpreter.getInstance() == null)
		{
//			IJ.error("Cannot install macro extensions from outside a macro!");
			return;
		}
		else
			Functions.registerExtensions(this);
	}

	private static void generateFTTCdata()
	{
		generateFTTCdata("Traction_" + file);
	}

	private static void generateFTTCdata(String outputFile)
	{
		int				i, j;
		int				nRow, nCol;
		long			startTime, elapsedTime;
		double			k2, k, gg, G0;
		int		[]		gridX, gridY;
		double	[]		Kx, Ky;
		double	[][]	disX, disY, TractionXR, TractionYR, TractionXF, TractionYF, TractionXN, TractionYN, dd, mag, magt, magN, magNt, TracXY, disXY = null;
		Matrix			TXY, G, H, G1, Gt, Ginv, dispXY, GtU, mmag, mmagt;
		DoubleFFT_2D	fftX, fftY;

		try
		{
			disXY = FTTC_plot.loadMatrixFromFile(dir + file, 4);
			// disXY is a 2D array with each row correspond to each vector, 4 columns correspond to x,y,dx,dy
		}
		catch (Exception e)
		{
			IJ.error(e.getMessage());
		}

		dim = FTTC_plot.getDimensions(disXY);

		// dim[0] is the number of data points in each row    (nx)
		// dim[1] is the number of data points in each column (ny)
		// dim[2] is the spacing between each data point      (in pixel)
		// dim[3] indicates whether it is row major (1) or column major(0)
		insetWidth    = (dim[0] - 1) * dim[2];
		insetHeight   = (dim[1] - 1) * dim[2];
		pictureWidth  =  dim[0]      * dim[2] + (int) (2 * disXY[0][0]);
		pictureHeight =  dim[1]      * dim[2] + (int) (2 * disXY[0][1]);

		if (!plugin)
			if (!getParams())
				return;

		disX = getSubMatrix	(disXY, dim[1], dim[0], 2, dim[3] == 1);
		// disX are 2D arrays with disX[x][y] = "the displacement in X direction at vector(x,y)"
		disY = getSubMatrix	(disXY, dim[1], dim[0], 3, dim[3] == 1);

		gridX = getCoord	(disXY, dim[1], dim[0], 0, dim[3] == 1);
		gridY = getCoord	(disXY, dim[1], dim[0], 1, dim[3] == 1);

		// convert distance in pixel into distance in meter (multiply by the pixel size)
		disX = scaleMatrix(disX, pixel_size);
		disY = scaleMatrix(disY, pixel_size);

		// get the distance between each data point (in pixel)
		distance_between_points_in_pixel = dim[2];

		// calculate the real distance between each data point (in meter)
		distance_between_points_in_m = distance_between_points_in_pixel * pixel_size;

		// record the original matrix size before padding and adding imaginary part
		nCol = disX[0]	.length;
		nRow = disX		.length;

		// start timer
		startTime = System.currentTimeMillis();

		try
		{
			disX = paddingZero(disX, nCol * 2, nRow);		 // padding 2 times in col for the imaginary part
			disY = paddingZero(disY, nCol * 2, nRow);
		}
		catch (Exception e)
		{
			IJ.error("Padding failed");
		}

		// convert disX, disY into fourier space
		fftX = new DoubleFFT_2D(nRow, nCol);
		fftY = new DoubleFFT_2D(nRow, nCol);
		fftX .realForwardFull(disX);
		fftY .realForwardFull(disY);

		// suppress the net translation by setting the first 0th element in fourier space to zero
		disX[0][0] = disX[0][1] = 0;
		disY[0][0] = disY[0][1] = 0;
		// disX is a complex matrix now, the [0][0] is the real part of the first element, [0][1] is the imaginary part of the first element

		// setup the wavevector array
		Kx = new double[nCol];
		Ky = new double[nRow];

		for (i = 0; i <= nCol / 2; i++)
			Kx[i]			=  (2 * Math.PI / distance_between_points_in_m) * ((double) i / (double) nCol);

		for (i = Math.round((float) nCol / 2 - 1); i > 0; i--)		// the negative frequency part of the wavevector
			Kx[nCol - i]	= -(2 * Math.PI / distance_between_points_in_m) * ((double) i / (double) nCol);

		for (i = 0; i <= nRow / 2; i++)
			Ky[i]			=  (2 * Math.PI / distance_between_points_in_m) * ((double) i / (double) nRow);

		for (i = Math.round((float) nRow / 2 - 1); i > 0; i--)		// the negative frequency part of the wavevector
			Ky[nRow - i]	= -(2 * Math.PI / distance_between_points_in_m) * ((double) i / (double) nRow);

		// create the empty matrix to store the calculated traction force
		TractionXF = new double[nRow][nCol * 2];
		TractionYF = new double[nRow][nCol * 2];
		// Matrix to store temporarily the calculated traction force
		TXY = new Matrix(2, 1);
		// Matrix of the Green function
		G = new Matrix(2, 2);
		// Identity matrix for 0th order regularization
		H = Matrix.identity(2, 2);
		// for 0th order regularization this term will be constant. See the equation (11) from: Sabass Biophy. J 2008
		H.timesEquals(lambda * lambda);

		for (j = 0; j < Ky.length; j++)
		{
			for (i = 0; i < Kx.length; i++)
			{
//				k	= Math.sqrt(Kx[i] * Kx[i] + Ky[j] * Ky[j]);
				k2	=           Kx[i] * Kx[i] + Ky[j] * Ky[j];
//				k	= Math.sqrt(k2);
				if (i == (float) nCol / 2 + 1 || j == (float) nRow / 2 + 1)
				{	// at the nyquist frequency, set the off-diagonal element to zero, see Butler et al. Am J Physil Cell Physiol 2001
					G		.set(0, 1, 0);
					G		.set(1, 0, 0);
				}
				else if (i != 0 || j != 0)
				{
					gg		= -mu * Kx[i] * Ky[j];
					G		.set(0, 1, -mu * Kx[i] * Ky[j]);
					G		.set(1, 0, -mu * Kx[i] * Ky[j]);
				}
				// the Green function
//				G0			= 2 * (1 + mu) / (E * Math.pow(k, 3));
//				G			.set(0, 0, (1 - mu) * (k * k) + mu * (Ky[j] * Ky[j]));
//				G			.set(1, 1, (1 - mu) * (k * k) + mu * (Kx[i] * Kx[i]));
				G0			= 2 * (1 + mu) / (E * Math.pow(k2, 1.5));
				G			.set(0, 0, (1 - mu) * k2      + mu * (Ky[j] * Ky[j]));
				G			.set(1, 1, (1 - mu) * k2      + mu * (Kx[i] * Kx[i]));
				G			.timesEquals	(G0);

				Gt			= G.transpose	();
				G1			= Gt.times		(G);
				G1			= G1.plus		(H);
				Ginv		= G1.inverse	();

				// to get the displacement
				dd			= new double[2][2];
				dd[0][0]	= disX[j][i * 2];
				dd[0][1]	= disX[j][i * 2 + 1];

				dd[1][0]	= disY[j][i * 2];
				dd[1][1]	= disY[j][i * 2 + 1];

				dispXY		= new Matrix(dd);
				GtU			= Gt.times(dispXY);
				TXY			= Ginv.times(GtU);

				// Store the calculated traction
				TractionXF[j][i * 2]		= TXY.get(0, 0);
				TractionXF[j][i * 2 + 1]	= TXY.get(0, 1);
				TractionYF[j][i * 2]		= TXY.get(1, 0);
				TractionYF[j][i * 2 + 1]	= TXY.get(1, 1);
			}
		}
		// set the net traction(first element in the matrix) to zero
		TractionXF[0][0] = 0;
		TractionXF[0][1] = 0;
		TractionYF[0][0] = 0;
		TractionYF[0][1] = 0;

		// inverse FFT of traction matrix to get the force in real space
		fftX.complexInverse(TractionXF, true);
		fftY.complexInverse(TractionYF, true);

		// To get the real part of the calculated Force
		TractionXR = real(TractionXF);
		TractionYR = real(TractionYF);

		// stop timer
		elapsedTime = System.currentTimeMillis() - startTime;

		IJ.showStatus		("FTTC done in " + elapsedTime + " msec");
		if(!plugin)
			showAnalysisParameters(elapsedTime, E, lambda, pixel_size);

		// draw the vector plot
		mag		= calcMagnitude(TractionXR, TractionYR);
		magt	= transpose(mag);

		// Calculate the data in pN (by multiplying the data in pPa by 1.0e6 x distance_between_points_in_m)
		window_area	= Math.pow(1.0e6 * distance_between_points_in_m, 2);
		TractionXN	= scaleMatrix(TractionXR, window_area);
		TractionYN	= scaleMatrix(TractionYR, window_area);
		magN		= scaleMatrix(mag		, window_area);
		magNt		= scaleMatrix(magt		, window_area);

		if(plotScalebar || plotVector || plotMagnitude || plotMagnitudeVector)
			plotDraw = true;
		else
			plotDraw = false;

		if (plotDraw)
		{
			TracXY = saveFTTCdata(gridX, gridY, TractionXR, TractionYR, mag, TractionXN, TractionYN, magN, dir + "Traction_" + file);
			// TracXY is 2D array organized like disXY, while TractionXR, Traction YR, mag are 2D arrays with array[x][y] correspond to vector(x,y)
			plot(TracXY, magt, magNt, file);
		}
		else
		{
			saveFTTCdata(gridX, gridY, TractionXR, TractionYR, mag, TractionXN, TractionYN, magN, dir + outputFile);
			FTTC_plot.max_vector_value_in_Pa = FTTC_plot.findMax2DArray(mag);
		}
		
		if(!plugin)
			saveFTTCparam(E, lambda, pixel_size, mu, dir + "FTTCparameters_" + file);
	}	

	public static void generateFTTC(String _dir, String _file, double _pixel_size, double _mu, double _E, double _lambda, int _unitIndex, int _pictureWidth, int _pictureHeight, boolean _plotScalebar, boolean _plotVector, boolean _plotMagnitude, boolean _plotMagnitudeVector, boolean _plotMagnitudeX, boolean _plotMagnitudeY, String _lut)
	{
		plugin				= true;
		dir					= _dir;
		file				= _file;
		pixel_size			= (1E-6) * _pixel_size;
		mu					= _mu;
		E					= _E;
		lambda				= _lambda;
		unitIndex			= _unitIndex;
		pictureWidth		= _pictureWidth;
		pictureHeight		= _pictureHeight;
		plotScalebar		= _plotScalebar;
		plotVector			= _plotVector;
		plotMagnitude		= _plotMagnitude;
		plotMagnitudeVector	= _plotMagnitudeVector;
		plotMagnitudeX		= _plotMagnitudeX;
		plotMagnitudeY		= _plotMagnitudeY;
		lut					= _lut;

		generateFTTCdata();
	}

	public static void generateFTTC(String _dir, String _file, String _output_file, double _pixel_size, double _mu, double _E, double _lambda, int _unitIndex, int _pictureWidth, int _pictureHeight, boolean _plotScalebar, boolean _plotVector, boolean _plotMagnitude, boolean _plotMagnitudeVector, boolean _plotMagnitudeX, boolean _plotMagnitudeY, String _lut)
	{
		plugin				= true;
		dir					= _dir;
		file				= _file;
		pixel_size			= (1E-6) * _pixel_size;
		mu					= _mu;
		E					= _E;
		lambda				= _lambda;
		unitIndex			= _unitIndex;
		pictureWidth		= _pictureWidth;
		pictureHeight		= _pictureHeight;
		plotScalebar		= _plotScalebar;
		plotVector			= _plotVector;
		plotMagnitude		= _plotMagnitude;
		plotMagnitudeVector	= _plotMagnitudeVector;
		plotMagnitudeX		= _plotMagnitudeX;
		plotMagnitudeY		= _plotMagnitudeY;
		lut					= _lut;

		generateFTTCdata(_output_file);
	}

	public static double getScaleValue(String _dir, String _file, double _pixel_size, double _mu, double _E, double _lambda, int _unitIndex, int _pictureWidth, int _pictureHeight, boolean _plotScalebar, boolean _plotVector, boolean _plotMagnitude, boolean _plotMagnitudeVector, boolean _plotMagnitudeX, boolean _plotMagnitudeY, String _lut)
	{
		plugin				= true;
		dir					= _dir;
		file				= _file;
		pixel_size			= (1E-6) * _pixel_size;
		mu					= _mu;
		E					= _E;
		lambda				= _lambda;
		unitIndex			= _unitIndex;
		pictureWidth		= _pictureWidth;
		pictureHeight		= _pictureHeight;
		plotScalebar		= _plotScalebar;
		plotVector			= _plotVector;
		plotMagnitude		= _plotMagnitude;
		plotMagnitudeVector	= _plotMagnitudeVector;
		plotMagnitudeX		= _plotMagnitudeX;
		plotMagnitudeY		= _plotMagnitudeY;
		lut					= _lut;

		generateFTTCdata();

		return FTTC_plot.max_vector_value_in_Pa;
	}
	
	public static double getScaleValue(String _dir, String _file, String _output_file, double _pixel_size, double _mu, double _E, double _lambda, int _unitIndex, int _pictureWidth, int _pictureHeight, boolean _plotScalebar, boolean _plotVector, boolean _plotMagnitude, boolean _plotMagnitudeVector, boolean _plotMagnitudeX, boolean _plotMagnitudeY, String _lut)
	{
		plugin				= true;
		dir					= _dir;
		file				= _file;
		pixel_size			= (1E-6) * _pixel_size;
		mu					= _mu;
		E					= _E;
		lambda				= _lambda;
		unitIndex			= _unitIndex;
		pictureWidth		= _pictureWidth;
		pictureHeight		= _pictureHeight;
		plotScalebar		= _plotScalebar;
		plotVector			= _plotVector;
		plotMagnitude		= _plotMagnitude;
		plotMagnitudeVector	= _plotMagnitudeVector;
		plotMagnitudeX		= _plotMagnitudeX;
		plotMagnitudeY		= _plotMagnitudeY;
		lut					= _lut;

		generateFTTCdata(_output_file);

		return FTTC_plot.max_vector_value_in_Pa;
	}

	private ExtensionDescriptor[] extensions =
	{
		ExtensionDescriptor.newDescriptor("getScaleValue", this),
	};

	public ExtensionDescriptor[] getExtensionFunctions()
	{
		return extensions;
	}

	public String handleExtension(String name, Object[] args)
	{
		if (name.equals("getScaleValue"))
			return String.valueOf(FTTC_plot.max_vector_value_in_Pa);
    
		return null;
	}

	private static double[][] calcMagnitude(double[][] arrayX, double[][] arrayY)
	{
		int				i, j;
		double	[][]	magnitude = new double[arrayX.length][arrayX[0].length];

		for (j = 0; j < arrayX.length; j++)
			for (i = 0; i < arrayX[0].length; i++)
				magnitude[j][i] = Math.sqrt(arrayX[j][i] * arrayX[j][i] + arrayY[j][i] * arrayY[j][i]);

		return magnitude;
	}

	private static double[][] transpose(double[][] array)
	{
		int				i, j;
		double	[][]	transposed = new double[array[0].length][array.length];

		for (j = 0; j < array.length; j++)
			for (i = 0; i < array[0].length; i++)
				transposed[i][j] = array[j][i];

		return transposed;
	}

	private static void plot(double[][] data, double mag[][], double magN[][], String title)
	{
		double plotColorScale, plotColorScaleN;
		ImageProcessor ip			= new ColorProcessor(pictureWidth, pictureHeight);
//		int[] plotDataDimensions	= FTTC_plot.getDimensions(data);

		FTTC_plot.loadLut(lut);
		FTTC_plot.max_vector_value_in_Pa	= FTTC_plot.findMax2DArray(mag);
		FTTC_plot.max_vector_value_in_N		= FTTC_plot.max_vector_value_in_Pa * window_area;
		plotColorScale						= 24 / FTTC_plot.max_vector_value_in_Pa;
//		plotColorScaleN						= 24 / FTTC_plot.max_vector_value_in_N;
		plotColorScaleN						= plotColorScale                   / window_area;

		FTTC_plot.plotData(title, pictureWidth, pictureHeight, insetWidth, insetHeight, FTTC_plot.colors, FTTC_plot.getCm(), dim, data, mag, magN, plotColorScale, plotColorScaleN, FTTC_plot.max_vector_value_in_Pa, FTTC_plot.max_vector_value_in_N, plotScalebar, plotVector, plotMagnitude, plotMagnitudeVector, plotMagnitudeX, plotMagnitudeX, unitIndex);

		return;
	}

	static public double[][] paddingZero(double[][] matrix, int sizeX, int sizeY) throws Exception
	{
		int				i, j;
		double	[][]	matP = new double[sizeY][sizeX];			//42x80	

		if (matrix.length > sizeY || matrix[0].length > sizeX) 
			throw new Exception("matrix larger than specified size");

		for (j = 0; j < matP.length; j++)				//42
			for (i = 0; i < matP[0].length; i++)		//80
				if (j < matrix.length && i < matrix[0].length)
					matP[j][i] = matrix[j][i];
				else
					matP[j][i] = 0.0;

		return matP;
	}

	private static double[][] real(double[][] compMat)
	{
		int				i, j;
		double	[][]	realMat = new double[compMat.length][compMat[0].length / 2];

		for (j = 0; j < realMat.length; j++)
			for (i = 0; i < realMat[0].length; i++)
				realMat[j][i] = compMat[j][i * 2];

		return realMat;
	}

	private static double[][] saveFTTCdata(int[] gridX, int[] gridY, double[][] tracX, double[][] tracY, double[][] mag, double[][] tracXN, double[][] tracYN, double[][] magN, String path)
	{
		int	i, j;
	
		if (gridX.length != tracX[0].length || gridX.length != tracY[0].length || gridY.length != tracX.length || gridY.length != tracY.length)
		{
			IJ.error("gridX.length: "		+ gridX.length);
			IJ.error("gridY.length: "		+ gridY.length);
			IJ.error("tracX.length: "		+ tracX.length);
			IJ.error("tracX[0].length: "	+ tracX[0].length);
			IJ.error("tracY.length: "		+ tracY.length);
			IJ.error("tracY[0].length: "	+ tracY[0].length);
			IJ.error("the dimension of FTTC output is not the same as grid");
			return null;
		}
		double[][] TXY = new double[tracX.length * tracX[0].length][8];

		StringBuilder sb = new StringBuilder();

		for (j = 0; j < tracX.length; j++)
		{
			for (i = 0; i < tracX[0].length; i++)
			{
				TXY[tracX[0].length * j + i][0] = gridX	[i];
				TXY[tracX[0].length * j + i][1] = gridY	[j];
				TXY[tracX[0].length * j + i][2] = tracX	[j][i];
				TXY[tracX[0].length * j + i][3] = tracY	[j][i];
				TXY[tracX[0].length * j + i][4] = mag	[j][i];
				TXY[tracX[0].length * j + i][5] = tracXN[j][i];
				TXY[tracX[0].length * j + i][6] = tracYN[j][i];
				TXY[tracX[0].length * j + i][7] = magN	[j][i];

				sb.append(String.valueOf(gridX	   [i]));
				sb.append(" ");
				sb.append(String.valueOf(gridY	[j]   ));
				sb.append(" ");
				sb.append(String.valueOf(tracX	[j][i]));
				sb.append(" ");
				sb.append(String.valueOf(tracY	[j][i]));
				sb.append(" ");
				sb.append(String.valueOf(mag	[j][i]));
				sb.append(" ");
				sb.append(String.valueOf(tracXN	[j][i]));
				sb.append(" ");
				sb.append(String.valueOf(tracYN	[j][i]));
				sb.append(" ");
				sb.append(String.valueOf(magN	[j][i]));
				sb.append("\n");
			}
		}

		String s = sb.toString();
		IJ.saveString(s, path);
		return TXY;
	}

	private static void saveFTTCparam(double _E, double _lambda, double _pixel_size, double _Poisson, String path)
	{
		StringBuilder sb = new StringBuilder();

		sb.append("Young's Modulus (E): ");
		sb.append(String.valueOf(_E));
		sb.append("\n");
		sb.append("Regularization factor(lambda): ");
		sb.append(String.valueOf(_lambda));
		sb.append("\n");
		sb.append("pixel size (in micron): ");
		sb.append(String.valueOf(_pixel_size));
		sb.append("\n");
		sb.append("Poisson ratio: ");
		sb.append(String.valueOf(_Poisson));
		sb.append("\n");

		String s = sb.toString();
		IJ.saveString(s, path);
		return;
	}

	private static double[][] getSubMatrix(double[][] matrix, int row, int col, int element, boolean rowMajor)
	{
		int				i, j;
		double	[][]	subM = new double[row][col];

		for (j = 0; j < row; j++)
		{
			for (i = 0; i < col; i++)
			{
				if (rowMajor)
					subM[j][i] = matrix[j * col + i][element];
				else
					subM[j][i] = matrix[i * row + j][element];
			}
		}

		return subM;
	}

	private static int[] getCoord(double[][] matrix, int row, int col, int element, boolean rowMajor)
	{
		int		i;
		int	[]	Coord;

		if (element == 0)				// getting x coordinates
		{
			Coord = new int[col];
			for (i = 0; i < col; i++)
				if (rowMajor)			// y was fixed first
					Coord[i] = (int) matrix[i]      [0];
				else
					Coord[i] = (int) matrix[i * row][0];
		}
		else if (element == 1)			// getting y coordinates
		{
			Coord = new int[row];
			for (i = 0; i < row; i++)
				if (rowMajor)			// y was fixed first
					Coord[i] = (int) matrix[i * col][1];
				else
					Coord[i] = (int) matrix[i]      [1];
		}
		else
		{
			IJ.error("Error in the definition of your element choice!");
			Coord = null;
		}

		return Coord;
	}

	private static double[][] scaleMatrix(double[][] mat, double scale)
	{
		int				i, j;
		double	[][]	sMat = new double[mat.length][mat[0].length];

		for (j = 0; j < mat.length; j++)
			for (i = 0; i < mat[0].length; i++)
				sMat[j][i] = mat[j][i] * scale;

		return sMat;
	}

	private static boolean getParams()
	{
		// generate the items of the lutText array
		int					i, lutIndex	= 0;
		String	[]			lutList, lutText;
		lutList				= IJ.getLuts();
		lutText				= new String[lutList.length + 2];
		lutText[0]			= "multitxt";
		lutText[1]			= "S_Pet";
		for(i = 0; i < lutList.length; i++)
			lutText[i + 2]	= lutList[i];
	
		GenericDialog gd = new GenericDialog("FTTC");

		gd.addNumericField	("Pixel_size"							, pixel_size * 1E6	, 3, 6, "m");
		gd.addNumericField	("Poisson_ratio_of_the_gel"				, mu				, 1, 6, "-");
		gd.addChoice		("Young's_modulus_of_the_gel"			, EText				, EText[EIndex]);
		gd.addChoice		("Regularization_factor"				, lambdaText		, lambdaText[lambdaIndex]);
		gd.addChoice		("Ouput_unit(s)"						, unitText			, unitText[unitIndex]);
		gd.addChoice		("                LUT_for_color_coding"	, lutText			, lutText[lutIndex]);
		gd.addNumericField	("Plot_width :"							, pictureWidth		, 0, 7, "pixels");
		gd.addNumericField	("Plot_height:"							, pictureHeight		, 0, 7, "pixels");
		gd.addCheckbox		("Draw_scale"							, true);
		gd.addCheckbox		("Draw_vector"							, true);
		gd.addCheckbox		("Draw_magnitude"						, true);
		gd.addCheckbox		("Draw_magnitude_and_vector"			, true);
		gd.addCheckbox		("Draw_X_magnitude"						, false);
		gd.addCheckbox		("Draw_Y_magnitude"						, false);

		// Put the LUT choices into focus
		Vector choices			= gd.getChoices();
		final Choice lutChoice	= (Choice)(choices.elementAt(3));
		gd.addWindowListener( new WindowAdapter()
		{
			public void windowOpened( WindowEvent e )
			{
				lutChoice.requestFocusInWindow();
			}
		}); 

		gd.showDialog();

		pixel_size			= (1E-6) *	gd.getNextNumber		();
		mu					=			gd.getNextNumber		();
//		E					=			gd.getNextNumber		();
		EIndex				=			gd.getNextChoiceIndex	();
		lambdaIndex			=			gd.getNextChoiceIndex	();
		unitIndex			=			gd.getNextChoiceIndex	();
		lutIndex			=			gd.getNextChoiceIndex	();
		pictureWidth		= (int)		gd.getNextNumber		();
		pictureHeight		= (int)		gd.getNextNumber		();
		plotScalebar		=			gd.getNextBoolean		();
		plotVector			=			gd.getNextBoolean		();
		plotMagnitude		=			gd.getNextBoolean		();
		plotMagnitudeVector	=			gd.getNextBoolean		();
		plotMagnitudeX		=			gd.getNextBoolean		();
		plotMagnitudeY		=			gd.getNextBoolean		();
		E					=			EValues[EIndex];
		lambda				=			lambdaValues[lambdaIndex];
		lut					=			lutText[lutIndex];

		if (gd.wasCanceled())
			return false;

		return true;
	}

	static public StringBuffer generatePIVToPrint(double[][] data)
	{
		int 			i, j;
		StringBuffer	info	= new StringBuffer();
		NumberFormat	nf		= NumberFormat.getInstance();
		nf.setMaximumFractionDigits(12);
		nf.setMinimumFractionDigits(12);

		for (j = 0; j < data.length; j++)
		{
			for (i = 0; i < data[0].length; i++)
			{
				info.append(nf.format(data[j][i]));
				info.append(" ");
			}
			info.append("\n");
		}

		return info;
	}

	static public StringBuffer generateArrayToPrint(double[] data)
	{
		int				i;
		StringBuffer	info	= new StringBuffer();
		NumberFormat	nf		= NumberFormat.getInstance();
		nf.setMaximumFractionDigits(12);
		nf.setMinimumFractionDigits(12);

		for (i = 0; i < data.length; i++)
		{
			info.append(nf.format(data[i]));
			info.append("\n");
		}

		return info;
	}
	
	private static void showAnalysisParameters(double elapsedTime, double E, double lambda, double pixel_size)
	{
		String			title;
		Frame			frame;
		TextWindow		tw;
		TextPanel		tp;
		NumberFormat	nf		= NumberFormat.getInstance();
		nf.setMaximumFractionDigits(5);
	
		title		= "Analysis parameters";
		frame		= WindowManager.getFrame(title);
		if (!(frame instanceof TextWindow))
		{
			IJ.run	("New... ", "name=[" + title + "] type=Table");
			frame	= WindowManager.getFrame(title);
		}
		tw			= (TextWindow) frame;
		tw			.setSize	(410, 200);
		tw			.setLocation( 10,   10);
		tp			= tw.getTextPanel();

		tp.clear				();
		tp.setColumnHeadings	("Parameter\tValue\tunit");
		tp.appendLine			("FTTC done in\t"			+ Math.round(elapsedTime)		+ "\tmsec");
		tp.appendLine			("E (Young's modulus)\t"	+ Math.round(E)					+ "\tPa");
		tp.appendLine			("Lambda\t"					+ lambda						+ "\t-");
		tp.appendLine			("Pixel size\t"				+ nf.format(pixel_size * 1E6)	+ "\tm");
	}
}