package mosaic.particleTracker;

import java.awt.Button;
import java.awt.Checkbox;
import java.awt.Choice;
import java.awt.Color;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Label;
import java.awt.Panel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.Vector;

import ij.IJ;
import ij.ImagePlus;
import ij.WindowManager;
import ij.gui.ImageWindow;
import ij.gui.Overlay;
import ij.gui.Plot;
import ij.gui.PlotWindow;
import ij.measure.ResultsTable;
import ij.process.ByteProcessor;

import mosaic.core.detection.Particle;
import mosaic.plugins.ParticleTracker3DModular_;
import mosaic.utils.ArrayOps;
import mosaic.utils.ArrayOps.MinMax;


/**
 *  This class is responsible for displaying MSS/MSD plots.
 *  @author Krzysztof Gonciarz <gonciarz@mpi-cbg.de>
 */
class TrajectoryAnalysisPlot extends ImageWindow implements ActionListener, ItemListener
{
	private static	final	long						serialVersionUID		= 1L;

	private			final	ParticleTracker3DModular_	particleTracker3DModular;

	// UI stuff
	private					Button				iMssButton;
	private					Button				iMsdButton;
	private					Button				iGetDataButton;
	private					Button				iGetAllDataButton;
	private					Checkbox			iLogScale;
	private					Choice				iTrajectoryChoice;

	private					Vector <Trajectory>	iTrajectories;
	private					TrajectoryAnalysis	iTrajectoryAnalysis;

	private					boolean				iMssPlot						= true;
	private					boolean				iGetAllData						= false;
	private					double				iPixelDim;
	private					double				iTimeInterval;
	private					double				iX[];
	private					double				iY[];
	private					String				iXname;
	private					String				iYname;
	private					ResultsTable		iTable							= null;

	// Dimensions of plot
	private static	final	int					WIN_WIDTH						= 820;
	private static	final	int					WIN_HEIGHT						= 600;

	/**
	 * @param aTrajectories	Trajectories to be analyzed and plotted.
	 * @param aPixelDim		Dimensions of pixel in meters (needed for proper trajectory analysis)
	 * @param aTimeInterval	Time interval between frames used to calculate trajectory analysis.
	 */
	public TrajectoryAnalysisPlot(ParticleTracker3DModular_ particleTracker3DModular_, Vector <Trajectory> aTrajectories, double aPixelDim, double aTimeInterval)
	{
		// A little bit nasty but working method of setting window size (and further plot size).
		// Other methods like setSize(...) do not work even if applied to both - ImageWindow and Plot
		super(new ImagePlus("Trajectory Analysis", new ByteProcessor(WIN_WIDTH, WIN_HEIGHT, new byte[WIN_WIDTH * WIN_HEIGHT], null)));

		particleTracker3DModular	= particleTracker3DModular_;
		iTrajectories				= aTrajectories;
		iPixelDim					= aPixelDim;
		iTimeInterval				= aTimeInterval;

		// Do all needed calculations to eliminate the trajectories that are not valid
		for(int trj = iTrajectories.size() - 1; trj >= 0; trj--)
		{
			// Calculate MSD / MSS
			iTrajectoryAnalysis	= new TrajectoryAnalysis	(iTrajectories.get(trj));
			iTrajectoryAnalysis	.setLengthOfAPixel			(iPixelDim);
			iTrajectoryAnalysis	.setTimeInterval			(iTimeInterval);
			if (iTrajectoryAnalysis.calculateAll() != TrajectoryAnalysis.SUCCESS)
				iTrajectories.remove(trj);
		}
		particleTracker3DModular.results_window.text_panel.appendLine(iTrajectories.size() + " Trajectories are left after elimination of the non valid ones");

		if(iTrajectories.size() == 0)
		{
			this.close();
			IJ.error("It is impossible to calculate MSS/MSD for any of the selected trajectory!");
			return;
		}
		else
		{
			// Calculate MSD / MSS
			iTrajectoryAnalysis	= new TrajectoryAnalysis	(iTrajectories.get(iTrajectories.size() - 1));
			iTrajectoryAnalysis	.setLengthOfAPixel			(iPixelDim);
			iTrajectoryAnalysis	.setTimeInterval			(iTimeInterval);
			if (iTrajectoryAnalysis.calculateAll() != TrajectoryAnalysis.SUCCESS)
			{
				this.close();
				IJ.error("It is impossible to calculate MSS/MSD for this trajectory!");
				return;
			}
			displayTrajectory(iTrajectories.get(iTrajectories.size() - 1).iSerialNumber);

			// Create UI for this window
			iMssButton			= new Button(" MSS ");
			iMssButton			.addActionListener(this);
			iMsdButton			= new Button(" MSD ");
			iMsdButton			.addActionListener(this);
			iGetDataButton		= new Button("Get Data");
			iGetDataButton		.addActionListener(this);
			iGetAllDataButton	= new Button("Get All Data");
			iGetAllDataButton	.addActionListener(this);
			iLogScale			= new Checkbox("logarithmic scale", true);
			iLogScale			.addItemListener(this);

			Panel panel = iTrajectories.size() == 1 ? new Panel(new GridLayout(1, 4)) : new Panel(new GridLayout(1, 7));

			panel.add(iMssButton);
			panel.add(iMsdButton);
			panel.add(iGetDataButton);
			panel.add(iLogScale);

			if (iTrajectories.size() > 1)
			{
				panel.add(new Label("Selected trajectory:", Label.RIGHT));
				iTrajectoryChoice = new Choice();
				for(int trj = iTrajectories.size() - 1; trj >= 0; trj--)
					iTrajectoryChoice.add(String.valueOf(iTrajectories.get(trj).iSerialNumber));
				iTrajectoryChoice.addItemListener(this);
				panel.add(iTrajectoryChoice);
				panel.add(iGetAllDataButton);
			}

			add(panel);

			// Show it all to user (by default it presents MSS/linear plot).
			plotMss();
			pack();
		}
	}

	/**
	 * @param aParticles Particles to be analyzed and plotted.
	 * @param aPixelDim Dimensions of pixel in meters (needed for proper trajectory analysis)
	 * @param aTimeInterval Time interval between frames used to calculate trajectory analysis.
	 */
	public TrajectoryAnalysisPlot(ParticleTracker3DModular_ particleTracker3DModular_, Trajectory aTrajectory, double aPixelDim, double aTimeInterval)
	{
		// A little bit nasty but working method of setting window size (and further plot size).
		// Other methods like setSize(...) do not work even if applied to both - ImageWindow and Plot
		super(new ImagePlus("Trajectory Analysis", new ByteProcessor(WIN_WIDTH, WIN_HEIGHT, new byte[WIN_WIDTH * WIN_HEIGHT], null)));

		particleTracker3DModular	= particleTracker3DModular_;

		// Create UI for this window
		iMssButton		= new Button(" MSS ");
		iMssButton		.addActionListener(this);
		iMsdButton		= new Button(" MSD ");
		iMsdButton		.addActionListener(this);
		iGetDataButton	= new Button("Get Data");
		iGetDataButton	.addActionListener(this);
		iLogScale		= new Checkbox("logarithmic scale", true);
		iLogScale		.addItemListener(this);

		final Panel panel = new Panel(new GridLayout(1, 4));
		panel.add(iMssButton);
		panel.add(iMsdButton);
		panel.add(iGetDataButton);
		panel.add(iLogScale);

		add(panel);

		// Do all needed calculations

		// Calculate MSD / MSS
		iTrajectoryAnalysis	= new TrajectoryAnalysis	(aTrajectory);
		iTrajectoryAnalysis	.setLengthOfAPixel			(aPixelDim);
		iTrajectoryAnalysis	.setTimeInterval			(aTimeInterval);
		if (iTrajectoryAnalysis.calculateAll() != TrajectoryAnalysis.SUCCESS)
		{
			this.close();
			IJ.error("It is impossible to calculate MSS/MSD for this trajectory!");
			return;
		}
		// Show it all to user (by default it presents MSS/linear plot).
		plotMss();
		pack();
	}

	/**
	 * This method updates plot according to given data. It presents data set of points
	 * altogether with straight line with given slope an y-axis intercept.
	 *
	 * @param aX data set with x points to be presented
	 * @param aY data set with y points to be presented
	 * @param aSlope slope of linear regression
	 * @param aY0 y-axis intercept of linear regression
	 * @param aXlabel
	 * @param aYlabel
	 * @param aWindowLabel title of window
	 */
	private void updatePlot(double[] aX, double[] aY, double aSlope, double aY0, Double aDiffusionCoefficient, Double aChiSquare, String aXlabel, String aYlabel, String aWindowLabel)
	{
		// Generate data for slop line
		final	double	[]		slopeLine	= new double[aX.length];
		for (int i = 0; i < aX.length; ++i)
			slopeLine[i] = aSlope * aX[i] + aY0;

		// Calculate X/Y min/max for plot
		final	double			minX		= aX[0];
		final	double			maxX		= aX[aX.length-1];
		
				MinMax<Double>	mmY			= ArrayOps.findMinMax(aY);
				double			minY		= mmY.getMin();
				double			maxY		= mmY.getMax();
		
				MinMax<Double>	mmSlopeLine	= ArrayOps.findMinMax(slopeLine);
								minY		= Math.min(minY, mmSlopeLine.getMin());
								maxY		= Math.max(maxY, mmSlopeLine.getMax());

		// Create plot with slope line
//		PlotWindow.noGridLines = false; // draw grid lines
//		final Plot plot = new Plot("", aXlabel, aYlabel, aX, slopeLine);	// Deprecated
		final Plot plot = new Plot("", aXlabel, aYlabel);
		plot.setSize(WIN_WIDTH, WIN_HEIGHT);
		setTitle(aWindowLabel);
		plot.setLimits(minX, maxX, minY, maxY);
		plot.setLineWidth(2);

		// Plot data
		plot.setColor	(Color.blue);
		plot.addPoints	(aX, slopeLine, Plot.LINE);
		plot.setColor	(Color.red);
		plot.addPoints	(aX, aY, Plot.X);

		// add label
		plot.setColor(Color.DARK_GRAY);
		plot.changeFont(new Font("Helvetica", Font.BOLD, 14));
		if (aDiffusionCoefficient == null)
		{
			plot.addLabel(0.05, 0.10, String.format("Slope            a   = %4.8f", aSlope));
			plot.addLabel(0.05, 0.15, String.format("Intercept      y0 = %5.8f", aY0));
			plot.addLabel(0.05, 0.20, String.format("Chi square \u03a7^2 = %5.8f", aChiSquare));
		}
		else
		{
			plot.addLabel(0.05, 0.10, String.format("slope                           a   = %4.8f", aSlope));
			plot.addLabel(0.05, 0.15, String.format("Intercept                     y0 = %5.8f", aY0));
			plot.addLabel(0.05, 0.20, String.format("Diffusion coefficient D2 = %5.8e", aDiffusionCoefficient));
			plot.addLabel(0.05, 0.25, String.format("Chi square                \u03c7^2 = %5.8f", aChiSquare));
		}

		// color for slope line
//		plot.setColor(Color.blue);
		setImage(plot.getImagePlus());
	}

	/**
	 * Plots MSS (moment scaling spectrum) with log/linear plot.
	 */
	private void plotMss()
	{
		iMssPlot = true;
		if (iLogScale.getState())
		{
			iX = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getMomentOrders());
			iY = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getGammasLogarithmic());
			iXname = "log(moment order \u03BD)";
			iYname = "log(scaling coefficient \u213D)";
			updatePlot(	iX,
						iY,
						iTrajectoryAnalysis.getMSSlogarithmic			(),
						iTrajectoryAnalysis.getMSSlogarithmicY0			(),
						null,
						iTrajectoryAnalysis.getMSSlogarithmicChiSquare	(),
						iXname, iYname, "MSS (log)");
		}
		else
		{
			iX = iTrajectoryAnalysis.toDouble(iTrajectoryAnalysis.getMomentOrders());
			iY = iTrajectoryAnalysis.getGammasLogarithmic();
			iXname = "moment order \u03BD";
			iYname = "scaling coefficient \u213D";
			updatePlot(	iX,
						iY,
						iTrajectoryAnalysis.getMSSlinear				(),
						iTrajectoryAnalysis.getMSSlinearY0				(),
						null,
						iTrajectoryAnalysis.getMSSlinearChiSquare		(),
						iXname, iYname, "MSS");
		}
	}

	/**
	 * Plots MSD (mean square displacement -> mean moment order = 2) with log/linear plot
	 */
	private void plotMsd()
	{
		iMssPlot = false;
		final int order = 1; // special case for order=2 -> MSD (array starts with 0 for moment=1)
		int size = iTrajectoryAnalysis.getFrameShifts().length;
		double[] timeSteps = new double[size];
		for (int i = 0; i < size; ++i)
			timeSteps[i] = iTrajectoryAnalysis.getFrameShifts()[i] * iTrajectoryAnalysis.getTimeInterval();
		if (iLogScale.getState())
		{
			iX = iTrajectoryAnalysis.toLogScale(timeSteps);
			iY = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getMSDforMomentIdx(order));
			iXname = "log(\u0394t)";
			iYname = "log(\u03bc(\u0394t))";
			updatePlot(	iX,
						iY,
						iTrajectoryAnalysis.getGammasLogarithmic			()	[order],
						iTrajectoryAnalysis.getGammasLogarithmicY0			()	[order],
						iTrajectoryAnalysis.getDiffusionCoefficients		()	[order],
						iTrajectoryAnalysis.getGammasLogarithmicChiSquare	()	[order],
						iXname, iYname, "MSD (log)");
		}
		else
		{
			iX = timeSteps;
			iY = iTrajectoryAnalysis.getMSDforMomentIdx(order);
			iXname = "\u0394t";
			iYname = "\u03BC(\u0394t)";
			updatePlot(	iX,
						iY,
						iTrajectoryAnalysis.getGammasLinear					()[order],
						iTrajectoryAnalysis.getGammasLinearY0				()[order],
						null,
						iTrajectoryAnalysis.getGammasLinearChiSquare		()[order],
						iXname, iYname, "MSD");
		}
	}

	private void updatePlot()
	{
		if (iTrajectories != null)
		{
			iTrajectoryAnalysis	= new TrajectoryAnalysis	(iTrajectories.get(iTrajectories.size() - 1 - iTrajectoryChoice.getSelectedIndex()));
			iTrajectoryAnalysis	.setLengthOfAPixel			(iPixelDim);
			iTrajectoryAnalysis	.setTimeInterval			(iTimeInterval);
			if (iTrajectoryAnalysis.calculateAll() != TrajectoryAnalysis.SUCCESS)
			{
				this.close();
				IJ.error("It is impossible to calculate MSS/MSD for this trajectory!");
				return;
			}
			displayTrajectory(iTrajectories.get(iTrajectories.size() - 1 - iTrajectoryChoice.getSelectedIndex()).iSerialNumber);
		}
		if (iMssPlot)
			plotMss();
		else
			plotMsd();
		if (WindowManager.getWindow("MSS/MSD plot data") != null)
		{
			getData();
			this.toFront();
		}
		iGetAllData = false;
	}

	private void displayTrajectory(int trajectoryNumber)
	{
		particleTracker3DModular.results_window.per_traj_label.setText("Trajectory " + trajectoryNumber);
		for (int i = 0; i < particleTracker3DModular.iTrajectories.size(); i++)
			if (particleTracker3DModular.iTrajectories.elementAt(i).iSerialNumber == trajectoryNumber)
			{
				WindowManager.getImage("All Trajectories Visual").getOverlay().clear();
				WindowManager.getImage("All Trajectories Visual").setRoi(particleTracker3DModular.iTrajectories.elementAt(i).trajectoryArea);
				return;
			}
	}

	private void displayAllChosenTrajectories()
	{
		Overlay traj_in_area_overlay = new Overlay();
		particleTracker3DModular.results_window.per_traj_label.setText("All chosen trajectories");

		for (int j = 0; j < iTrajectories.size(); j++)
			for (int i = 0; i < particleTracker3DModular.iTrajectories.size(); i++)
				if (particleTracker3DModular.iTrajectories.elementAt(i) == iTrajectories.elementAt(j))
				{
					traj_in_area_overlay.add(particleTracker3DModular.iTrajectories.elementAt(i).trajectoryArea);
					j++;
					break;
				}
		WindowManager.getImage("All Trajectories Visual").getOverlay().clear();
		WindowManager.getImage("All Trajectories Visual").setOverlay(traj_in_area_overlay);
	}

	private void getData()
	{
		if (iTable == null)
			iTable = new ResultsTable();
		iTable.reset();
		if (iTable != null)
		{
			for (int i = 1; i < 3; i++)
				iTable.setDecimalPlaces(i, 8);
			for (int i = 0; i < iX.length; ++i)
			{
				iTable.setValue(iXname, i, iX[i]);
				iTable.setValue(iYname, i, iY[i]);
			}
			iTable.show("MSS/MSD plot data");
		}
	}

	private void getAllData()
	{
		displayAllChosenTrajectories();
		if (WindowManager.getWindow("MSS/MSD plot data") == null)
			iTable = new ResultsTable();
		iTable.reset();
		if (iTable != null)
		{
			iTable.setNaNEmptyCells(true);
			final	int order = 1; // special case for order=2 -> MSD (array starts with 0 for moment=1)
					int iSize = 0;
			for (int i = 1; i < iTrajectories.size(); i++)
				iTable.setDecimalPlaces(i, 8);
			for(int trj = iTrajectories.size() - 1; trj >= 0; trj--)
			{
				// Calculate MSD / MSS
				iTrajectoryAnalysis	= new TrajectoryAnalysis	(iTrajectories.get(trj));
				iTrajectoryAnalysis	.setLengthOfAPixel			(iPixelDim);
				iTrajectoryAnalysis	.setTimeInterval			(iTimeInterval);
				if (iTrajectoryAnalysis.calculateAll() == TrajectoryAnalysis.SUCCESS)
				{
					if(iMssPlot)
					{
//						iSize = iX.length;			// to uncoment in order to get rid of the NaN within the MSS plot
						if (iLogScale.getState())
						{
							iX = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getMomentOrders());
							iY = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getGammasLogarithmic());
						}
						else
						{
							iX = iTrajectoryAnalysis.toDouble(iTrajectoryAnalysis.getMomentOrders());
							iY = iTrajectoryAnalysis.getGammasLogarithmic();
						}
					}
					else
					{
						if (iSize < iTrajectoryAnalysis.getFrameShifts().length)
							iSize = iTrajectoryAnalysis.getFrameShifts().length;
						double[] timeSteps = new double[iTrajectoryAnalysis.getFrameShifts().length];
						for (int i = 0; i < iTrajectoryAnalysis.getFrameShifts().length; ++i)
							timeSteps[i] = iTrajectoryAnalysis.getFrameShifts()[i] * iTrajectoryAnalysis.getTimeInterval();
						if (iLogScale.getState())
						{
							iX = iTrajectoryAnalysis.toLogScale(timeSteps);
							iY = iTrajectoryAnalysis.toLogScale(iTrajectoryAnalysis.getMSDforMomentIdx(order));
						}
						else
						{
							iX = timeSteps;
							iY = iTrajectoryAnalysis.getMSDforMomentIdx(order);
						}
					}
					for (int i = 0; i < iX.length; ++i)
					{
						if(trj == iTrajectories.size() - 1 || iSize == iTrajectoryAnalysis.getFrameShifts().length)
							iTable.setValue(iXname, i, iX[i]);
						iTable.setValue(iYname + " - " + iTrajectories.get(trj).iSerialNumber, i, iY[i]);
					}
				}
			}
			for(int trj = iTrajectories.size() - 1; trj >= 0; trj--)
			{
				for (int i = 0; i < iSize; ++i)
					if (Double.isNaN(iTable.getValue(iYname + " - " + iTrajectories.get(trj).iSerialNumber, i)))
						iTable.setValue(iYname + " - " + iTrajectories.get(trj).iSerialNumber, i, "");
			}
		}
		iTable.show("MSS/MSD plot data");
		iGetAllData = true;
	}

	@Override
	public void actionPerformed(ActionEvent ae)
	{
		final Object o = ae.getSource();

				if (o == iMssButton)
		{
			if (iGetAllData)
				updatePlot();
			plotMss();
			if (WindowManager.getWindow("MSS/MSD plot data") != null)
			{
				getData();
				this.toFront();
			}
		}
		else	if (o == iMsdButton)
		{
			if (iGetAllData)
				updatePlot();
			plotMsd();
			if (WindowManager.getWindow("MSS/MSD plot data") != null)
			{
				getData();
				this.toFront();
			}
		}
		else	if (o == iGetDataButton)
		{
			if (iGetAllData)
				updatePlot();
			getData();
		}
		else	if (o == iGetAllDataButton)
			getAllData();
	}

	@Override
	public void itemStateChanged(ItemEvent ie)
	{
		updatePlot();
	}
}