package mosaic.core.particleLinking;

import java.io.BufferedReader;
import java.io.FileReader;

import java.text.NumberFormat;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import org.apache.log4j.Logger;

import ij.IJ;

import mosaic.core.detection.Particle;


public abstract class ParticleLinker
{
	private static final Logger logger = Logger.getLogger(ParticleLinker.class);

	// PIV file parameters
	private		int 		plotW = 0;
	private		int 		plotH = 0;
	private		int			pivW, pivH;
	private		int  []		dimensions;		// record data dimensions as array (nColumn, nRow, vectorSpace)
	private		float[][]	magArr;
	private		float[][]	data;

	/**
	 * Second phase of the algorithm - <br>
	 * Identifies points corresponding to the
	 * same physical particle in subsequent frames and links the positions into trajectories <br>
	 * The length of the particles next array will be reset here according to the current linkrange
	 */
	public boolean linkParticles(List<Vector<Particle>> aParticles, LinkerOptions aLinkOpts)
	{
		logger.info("Linking options:" +
					", linkRange: "									+ aLinkOpts.linkRange +
					", maxDisplacement: "							+ aLinkOpts.maxDisplacement +
					", force: "										+ aLinkOpts.force +
					", straightLine: "								+ aLinkOpts.straightLine +
					", minSquaredDisplacementForAngleCalculation: "	+ aLinkOpts.minSquaredDisplacementForAngleCalculation +
					", lSpace: "									+ aLinkOpts.lSpace +
					", lFeature: "									+ aLinkOpts.lFeature +
					", lDynamic: "									+ aLinkOpts.lDynamic +
					", usePivFile: "								+ aLinkOpts.usePivFile +
					", filePivDirectoryAndName: "					+ aLinkOpts.filePivDirectoryAndName);

		// Reading the PIV file
		if(aLinkOpts.usePivFile)
		{
			try
			{
				data = load2DArrayFromFile(aLinkOpts.filePivDirectoryAndName);
			}
			catch (Exception e)
			{
				IJ.error(e.getMessage());
			}

			// get the data dimentsion and set the plot dimension
			dimensions = getDimensions(data);
			// dimensions[0] is the number of data points in each row    (nx)
			// dimensions[1] is the number of data points in each column (ny)
			// dimensions[2] is the spacing between each data point (in pixel)

			// Calculate the piv width and height, as well as the plot width and height
			pivW = (dimensions[0] - 1) * dimensions[2];
			pivH = (dimensions[1] - 1) * dimensions[2];
//			if (plotH == 0 && plotW == 0)
//			{
//				plotW = dimensions[0] * dimensions[2] + (int) (2 * data[0][0]);
//				plotH = dimensions[1] * dimensions[2] + (int) (2 * data[0][1]);
//			}

			// calculate the magnitude(norm of the vector)
//			magArr = get2DElement(data, dimensions[0], dimensions[1], 4);
		}

		final	int NumOfFrames	= aParticles.size();
				int LinkRange	= aLinkOpts.linkRange;
		for (int currFrame = 0; currFrame < NumOfFrames; ++currFrame)
		{
			for (Particle p : aParticles.get(currFrame))
			{
				p.lx  = p.ly  = p.lz = 0;
				p.lxa = p.lya = p.lza = 0;
			}
		}
		for (int currFrame = 0; currFrame < NumOfFrames; ++currFrame)
		{
			int currLinkRange = (currFrame < NumOfFrames - LinkRange) ? LinkRange : NumOfFrames - currFrame - 1;
			logInfo("----- Linking Frame "	+ (currFrame + 1) +
					"/"						+ NumOfFrames +
					" linkRange: "			+ currLinkRange +
					" ----------------------");

			Vector<Particle> p1 = aParticles.get(currFrame);
			int numOfParticles = p1.size();
			initParticlesLinkData(LinkRange, p1);

			for (int currLinkLevel = 1; currLinkLevel <= currLinkRange; ++currLinkLevel)
			{
				Vector<Particle> p2 = aParticles.get(currFrame + currLinkLevel);
				int numOfLinkParticles = aParticles.get(currFrame + (currLinkLevel )).size();
				
				final float maxCost = (float) Math.pow(currLinkLevel * aLinkOpts.maxDisplacement, 2);

				link(p1, p2, aLinkOpts, currFrame, NumOfFrames, numOfParticles, numOfLinkParticles, currLinkLevel, maxCost);
			}
		}

		return true;
	}

	protected abstract void link(Vector<Particle> p1, Vector<Particle> p2, LinkerOptions aLinkOpts, int currFrame, final int NumOfFrames, int numOfParticles, int numOfLinkParticles, int currLinkLevel, final float maxCost);

	public float linkCost(Particle pA, Particle pB, LinkerOptions aLinkOpts, int aLinkLevel)
	{
		float dx			= aLinkOpts.usePivFile	?	pB.getX() - pA.getX() - getDataValue(getDataIndex(pB.getX(), pB.getY()), 2) : pB.getX() - pA.getX();
		float dy			= aLinkOpts.usePivFile	?	pB.getY() - pA.getY() - getDataValue(getDataIndex(pB.getX(), pB.getY()), 3) : pB.getY() - pA.getY();
		float dz			=							pB.getZ() - pA.getZ();
		float distanceSq	= dx * dx + dy * dy + dz * dz;

		float dm0			= pB.getM0() - pA.getM0();
		float dm2			= pB.getM2() - pA.getM2();
		float momentsDist	= (float) Math.cbrt(dm0 * dm0 + dm2 * dm2);

		float linkCost		= distanceSq * aLinkOpts.lSpace + momentsDist * aLinkOpts.lFeature;

		if (aLinkOpts.force && pA.distance >= 0.0)
		{
			final float lx	= dx / aLinkLevel - pA.lx;
			final float ly	= dy / aLinkLevel - pA.ly;
			final float lz	= dz / aLinkLevel - pA.lz;

			final float dynamicCost = lx * lx + ly * ly + lz * lz;
			linkCost		+= aLinkOpts.lDynamic * dynamicCost;
		}
		else if (aLinkOpts.straightLine && pA.distance >= 0.0)
		{
			float lx2		= dx + pA.lxa;
			float ly2		= dy + pA.lya;
			float lz2		= dz + pA.lza;
			final float l2_m = (float) Math.sqrt(  lx2 *   lx2 +   ly2 *   ly2 +   lz2 *   lz2);
			final float l1_m = (float) Math.sqrt(pA.lx * pA.lx + pA.ly * pA.ly + pA.lz * pA.lz);

			if (l2_m >= aLinkOpts.minSquaredDisplacementForAngleCalculation && l1_m > 0)
			{
				final float lx1 = pA.lx / l1_m;
				final float ly1 = pA.ly / l1_m;
				final float lz1 = pA.lz / l1_m;
				
				lx2 /= l2_m;
				ly2 /= l2_m;
				lz2 /= l2_m;

				final float cosPhi = lx1 * lx2 + ly1 * ly2 + lz1 * lz2;
				linkCost += (cosPhi - 1) * (cosPhi - 1) * aLinkOpts.maxDisplacement * aLinkOpts.maxDisplacement;
			}
		}

		return linkCost;
	}

	protected void handleCostFeatures(Particle pA, Particle pB, LinkerOptions aLinkOpts, int aLinkLevel)
	{
		float dx = pB.getX() - pA.getX();
		float dy = pB.getY() - pA.getY();
		float dz = pB.getZ() - pA.getZ();

		if (aLinkOpts.force)
		{
			// Store the normalized linking vector
			pB.lx = dx / aLinkLevel;
			pB.ly = dy / aLinkLevel;
			pB.lz = dz / aLinkLevel;

			// We do not use distance is just to indicate that the particle has a link vector
			pB.distance = 1.0f;
		}
		else if (aLinkOpts.straightLine)
		{
			float distanceSq = dx*dx + dy*dy + dz*dz;
			if (distanceSq >= aLinkOpts.minSquaredDisplacementForAngleCalculation)
			{
				pB.lx = dx + pA.lxa;
				pB.ly = dy + pA.lya;
				pB.lz = dz + pA.lza;
			}
			else
			{
				// Propagate the previous link vector
				pB.lx = pA.lx;
				pB.ly = pA.ly;
				pB.lz = pA.lz;

				pB.lxa += dx + pA.lxa;
				pB.lya += dy + pA.lya;
				pB.lza += dz + pA.lza;
				float lengthSq = pB.lxa * pB.lxa + pB.lya * pB.lya + pB.lza * pB.lza;
				
				if (lengthSq >= aLinkOpts.minSquaredDisplacementForAngleCalculation)
				{
					pB.lx = pB.lxa;
					pB.ly = pB.lya;
					pB.lz = pB.lza;

					pB.lxa = 0;
					pB.lya = 0;
					pB.lza = 0;
				}
			}
			pB.distance = (float) Math.sqrt(distanceSq);
		}
	}

	protected void initParticlesLinkData(final int aLinkRange, Vector<Particle> aParticles)
	{
		for (Particle p : aParticles)
		{
			p.special		= false;
			p.next			= new int[aLinkRange];
			for (int n = 0; n < aLinkRange; ++n)
				p.next[n]	= -1;
		}
	}

//	private void logInfo(String aLogStr)		// to declare this method as private requires to declare them again within the inherited classes
	protected void logInfo(String aLogStr)
	{
		IJ.showStatus	(aLogStr);
		logger.info		(aLogStr);
	}

	/**
	 * The following methods come from the PIV plot_ plugin and are used in order to read the PIV data
	 */
	private float[][] load2DArrayFromFile(String path) throws Exception
	{
		ArrayList<String[]>	aL		= new ArrayList<String[]>(); 	// an array list to hold all vector info as String[]
		String	[]			cell;
		String				line;
		int row						= 0;
		float	[][]		matrix;
//		NumberFormat		nf		= NumberFormat.getInstance();
		NumberFormat		nf		= NumberFormat.getInstance(Locale.US);

		try
		{
			// open the file
			BufferedReader	r		= new BufferedReader(new FileReader(path));
			row = 0;
			while ((line = r.readLine()) != null)
			{
				line = line.trim();
				aL.add(line.split("\\s+"));	 // each element of the array list: "line" is actually a string[4]
				row++;
			}
			r.close();
		}
		catch (Exception e)
		{
			//IJ.error(e.getMessage());
			throw new Exception("Unsupported file format.");
		}

		matrix						= new float[row][5];
		Iterator <String[]> iter	= aL.iterator();
		int counter					= 0;

		// go over all elements of aL
		while (iter.hasNext())
		{
//			cell = (String[]) iter.next();
			cell = iter.next();			// cell correspond to one element(string[4]) of aL, that is: one entry of data containing x,y,ux,uy, L
			//check if we have the fifth column which correspond to the vector magnitude in our data.
			if (cell.length > 4)
			{
				for (int i = 0; i < 5; i++)
					matrix[counter][i] = nf.parse(cell[i]).floatValue();
				counter++;
			}
			else if (cell.length == 4)
			{
				for (int i = 0; i < 4; i++)
					matrix[counter][i] = nf.parse(cell[i]).floatValue();
				matrix[counter][4] = (float) Math.sqrt(matrix[counter][2] * matrix[counter][2] + matrix[counter][3] * matrix[counter][3]);
				counter++;
			}
			else
				throw new Exception("The file must have at least first 4 column: x,y,dx,dy separated by space or tab");
		}
		return matrix;
	}

	private int[] getDimensions(float[][] array)
	{
		int[] dm = new int[3];					// dm[2] is the spacing of vector (points), dm[0] is the number of points in x, dm[1] is np in y

		if ((array[1][0] - array[0][0]) == 0)	// x was fixed first
		{
			dm[2] = (int) (array[1][1] - array[0][1]);
			for (int i = 1; i < array.length; i++)
			{
				if (array[i][0] == array[i - 1][0])
					dm[1]++;
				else
					break;
			}
			dm[1]++;
			dm[0] = array.length / dm[1];
		}
		else
		{
			dm[2] = (int) (array[1][0] - array[0][0]);
			for (int i = 1; i < array.length; i++)
			{
				if (array[i][1] == array[i - 1][1])
					dm[0]++;
				else
					break;
			}
			dm[0]++;
			dm[1] = array.length / dm[0];
		}
		return dm;
	}

	/**
	 * extract one element from a 2D array [nData points][nElements]
	 * into another 2D data array with this specified element indexed by
	 * [nData points in x][nData points in y]
	*/
	private float[][] get2DElement(float[][] _piv, int nx, int ny, int nEle)
	{
		// check if the _piv array is fixing x first or y first
		boolean fx = _piv[0][0] == _piv[1][0] ? true : false;

		float[][] new2D = new float[nx][ny];
		for (int j = 0; j < ny; j++)
			for (int i = 0; i < nx; i++)
				new2D[i][j] = fx ? _piv[ny * i + j][nEle] : _piv[nx * j + i][nEle];

		return new2D;
	}

	/**
	 * Determine which index from the 2D data array should be used for further calculation
	*/
	private int getDataIndex(float posX, float posY)		// PosX and PosY are inversed since X and Y are inversed in the ParticleTracker code
	{
		if((posX > data[0][0] && posX <= pivW) && (posY > data[0][1] && posY <= pivH))
			return (int) (Math.round((posY - data[0][1]) / dimensions[2]) * dimensions[0] + Math.round((posX - data[0][0]) / dimensions[2]));
//			return (int) (Math.round((posX - data[0][1]) / dimensions[2]) * dimensions[0] + Math.round((posY - data[0][0]) / dimensions[2]));
		else
			return -1;
	}

	private float getDataValue(int index1, int index2)
	{
		if(index1 != -1)
			return data[index1][index2];
		else
			return 0.0f;
	}
}