package mosaic.particleTracker;


import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Arrays;
//import java.util.Iterator;
import java.util.ListIterator;
import java.util.Vector;

import ij.IJ;
import ij.ImagePlus;
import ij.gui.ImageCanvas;
import ij.gui.Roi;
import ij.gui.StackWindow;
import ij.measure.Calibration;
import mosaic.core.detection.MyFrame;
import mosaic.core.detection.MyFrame.DrawType;
import mosaic.core.detection.Particle;
import mosaic.plugins.ParticleTracker3DModular_;
import net.imglib2.img.Img;
import net.imglib2.type.numeric.ARGBType;


/**
 * Class that visualize a window with trajectory + insert a mouse listener to select trajectory
 *
 * @author Pietro Incardona
 */
public class TrajectoryStackWin extends StackWindow implements KeyListener, MouseListener
{
	private final				ParticleTracker3DModular_	particleTracker3DModular;
	private static final		long						serialVersionUID			= 1L;
	private						Button						drawTrajectories;
	private						Button						drawParticles;
	private						Button						filter_length;
	private final				Label						numberOfParticlesLabel;
	private						Img		<ARGBType>			out;
								int							lastState;
	private						Particle[]					user_traj_particles_array;
	private static final		String						HideTrajectories			= "Hide trajectories";
	private static final		String						ShowTrajectories			= "Show trajectories";
	private static final		String						HideParticles				= "Hide particles";
	private static final		String						ShowParticles				= "Show particles";

	/**
	 * Constructor. <br>
	 * Creates an instance of TrajectoryStackWindow from a given <code>ImagePlus</code> and <code>ImageCanvas</code> and a creates GUI panel. <br>
	 * Adds this class as a <code>MouseListener</code> to the given <code>ImageCanvas</code>
	 * 
	 * @param aimp
	 * @param icanvas
	 * @param particleTracker3DModular_
	 */
	public TrajectoryStackWin(ParticleTracker3DModular_ particleTracker3DModular_, ImagePlus aimp, ImageCanvas icanvas, Img<ARGBType> out_)
	{
		super							(aimp, icanvas);
		particleTracker3DModular		= particleTracker3DModular_;
		numberOfParticlesLabel			= new Label("");
		icanvas.addMouseListener		(this);
		icanvas.addKeyListener			(this);
		addPanel						();
		changeParticleNumberLabel		();
		out								= out_;
		user_traj_particles_array		= new Particle[particleTracker3DModular.getNumberOfFrames()];
	}

	private void changeParticleNumberLabel()
	{
		int currentframe = this.getImagePlus().getSlice() - 1;

		// understand the dimansionality
		final int nslices = this.getImagePlus().getNSlices();
		final int nframes = this.getImagePlus().getNFrames();

		// check the dimensionality
		if (nslices == 1)
			currentframe = this.getImagePlus().getChannel();		// 2D
		else if (nframes == 1)
			currentframe = this.getImagePlus().getSlice();			// 3D

		numberOfParticlesLabel.setText("Frame " + currentframe + ": " + particleTracker3DModular.iFrames[currentframe - 1].getParticles().size() + " particles");
	}

	@Override
	public String createSubtitle()
	{
		// overrided to get the right moment to update the label.
		changeParticleNumberLabel();
		return super.createSubtitle();
	}

	/**
	 * Adds a Panel with filter options button and number of particles in it to this window
	 */
	private void addPanel()
	{
		final Panel panel	= new Panel(new GridLayout(2, 2));

		String drawString1	= ShowTrajectories;
		System.out.println("INIT STATE: " + particleTracker3DModular.iTrajectories.get(0).drawTrajectory());
		if (particleTracker3DModular.iTrajectories.get(0).drawTrajectory())
			drawString1		= HideTrajectories;
		drawTrajectories	= new Button(drawString1);
		drawTrajectories	.addActionListener(this);
		panel				.add(drawTrajectories);

		filter_length		= new Button(" Filter Options ");
		filter_length		.addActionListener(this);
		panel				.add(filter_length);

		String drawString2	= ShowParticles;
		System.out.println("INIT STATE: " + particleTracker3DModular.iTrajectories.get(0).drawParticle());
		if (particleTracker3DModular.iTrajectories.get(0).drawParticle())
			drawString2		= HideParticles;
		drawParticles		= new Button(drawString2);
		drawParticles		.addActionListener(this);
		panel				.add(drawParticles);

		panel				.add(numberOfParticlesLabel);

		add(panel);

		pack();

		final Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
		final Point loc = getLocation();
		final Dimension size = getSize();
		if (loc.y + size.height > screen.height)
			getCanvas().zoomOut(0, 0);
	}

	/**
	 * Defines the action taken upon an <code>ActionEvent</code> triggered from buttons that have class <code>TrajectoryStackWindow</code> as their action listener: <br>
	 * <code>Button filter_length</code>
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public synchronized void actionPerformed(ActionEvent e)
	{
		final Object b = e.getSource();
		if (b == filter_length)
		{
			// if user cancelled the filter dialog - do nothing
			if (!particleTracker3DModular.filterTrajectories())
				return;
		}
		if (b == drawTrajectories)
		{
			// if user cancelled the filter dialog - do nothing
			System.out.println("NAME: " + drawTrajectories.getLabel());
			if (drawTrajectories.getLabel().equals(ShowTrajectories))
			{
				System.out.println("showing trajectories");
				drawTrajectories.setLabel(HideTrajectories);
				particleTracker3DModular.setDrawingTrajectory(true);
			}
			else
			{
				System.out.println("hiding trajectories");
				drawTrajectories.setLabel(ShowTrajectories);
				particleTracker3DModular.setDrawingTrajectory(false); 
			}
		}
		if (b == drawParticles)
		{
			// if user cancelled the filter dialog - do nothing
			System.out.println("NAME: " + drawParticles.getLabel());
			if (drawParticles.getLabel().equals(ShowParticles))
			{
				System.out.println							("showing particles");
				drawParticles.setLabel						(HideParticles);
				particleTracker3DModular.setDrawingParticle	(true);
			}
			else
			{
				System.out.println							("hiding particles");
				drawParticles.setLabel						(ShowParticles);
				particleTracker3DModular.setDrawingParticle	(false); 
			}
		}

		System.out.println("SOURCE: " + e.getSource());

		// Regenerate the image
		out = particleTracker3DModular.createHyperStackFromFrames();

		// generate an updated view with the ImagePlus in this window according to the new filter
		particleTracker3DModular.generateView(this.imp, this.out);
	}

	static public Roi getBiggerROI(Roi r, ImagePlus img, int size)
	{
		Rectangle bounds = r.getBounds();
		int newX = Math.max(0, bounds.x - size);
		int newY = Math.max(0, bounds.y - size);

		int	newWidth = bounds.width + 2 * size;
		if (newX + newWidth >= img.getWidth())
			newWidth = img.getWidth() - newX;
		int	newHeight = bounds.height + 2 * size;
		if (newY + newHeight >= img.getHeight())
			newHeight = img.getHeight() - newY;

		return new Roi(newX, newY, newWidth, newHeight);
	}

	/**
	 * Calulates the dilation mask
	 * <code>mask</code> is a var of class ParticleTracker_ and its modified internally here
	 * Adapted from Ingo Oppermann implementation
	 * @param mask_radius the radius of the mask (user defined)
	 */
	private static int calculateMask(int position_i, int position_j, int mask_radius)
	{
		if((position_i * position_i) + (position_j * position_j) <= mask_radius * mask_radius)
			return 1;
		else
			return 0;
	}

	/**
	 * Defines the action taken upon an <code>MouseEvent</code> triggered by left-clicking the mouse anywhere in this <code>TrajectoryStackWindow</code>
	 * 
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 */
	@Override
	public synchronized void mousePressed(MouseEvent e)
	{
		/* Reset selected trajectory */
		if (particleTracker3DModular.chosen_traj != -1)
		{
			final Vector<Trajectory> v = new Vector<Trajectory>();
			v.add(particleTracker3DModular.iTrajectories.get(particleTracker3DModular.chosen_traj));

			// If input is taken from csv files then iInputImage might not be there
			final Calibration cal = particleTracker3DModular.iInputImage == null ? null : particleTracker3DModular.iInputImage.getCalibration();
			MyFrame.updateImage(out, v, cal, DrawType.TRAJECTORY_HISTORY, particleTracker3DModular.getRadius());

			particleTracker3DModular.chosen_traj = -1;
		}
		/* get the coordinates of mouse while it was clicked */
		final	int							mouseX				= e.getX();
		final	int							mouseY				= e.getY();
		/* covert them to offScreen coordinates using the ImageCanvas of this window */
		final	int							offscreenX			= this.ic.offScreenX(mouseX);
		final	int							offscreenY			= this.ic.offScreenY(mouseY);
				boolean						trajectory_clicked	= false;
		final 	ListIterator <Trajectory>	iter				= particleTracker3DModular.iTrajectories.listIterator();

		/* Get pixel color */
		if (this.imp == null)
		{
			// there is not image this listener is dead remove it
			removeMouseListener(this);
			return;
		}

		final int cl[] = this.imp.getPixel(offscreenX, offscreenY);

		/* find the best Trajectory to match the mouse click */
		int ct = 0;
		while (iter.hasNext())
		{
			final Trajectory curr_traj = iter.next();

			if (curr_traj.color.getRed() == cl[0] && curr_traj.color.getGreen() == cl[1] && curr_traj.color.getBlue() == cl[2])
			{
				// Select given trajectory
				final Vector<Trajectory> v = new Vector<Trajectory>();
				v.add(curr_traj);

				// If input is taken from csv files then iInputImage might not be there
				final Calibration cal = particleTracker3DModular.iInputImage == null ? null : particleTracker3DModular.iInputImage.getCalibration();

				MyFrame.updateImage(out, v, cal, DrawType.TRAJECTORY_HISTORY, particleTracker3DModular.getRadius());
				trajectory_clicked = true;
				particleTracker3DModular.chosen_traj = ct;

				break;
			}

			ct++;
		}

		if (trajectory_clicked)
		{
			/* focus or mark the selected Trajectory according the the type of mouse click */
			this.imp.killRoi();
			this.imp.updateImage();
			// show the number of the selected Trajectory on the per trajectory panel in the results window
			particleTracker3DModular.results_window.per_traj_label.setText("Trajectory " + (particleTracker3DModular.chosen_traj + 1));
			if (e.getClickCount() == 2)
			{
				// "double-click"
				// Set the ROI to the trajectory focus_area
				Roi mouse_selection_area = (particleTracker3DModular.iTrajectories.elementAt(particleTracker3DModular.chosen_traj)).trajectoryArea;
				Roi roi = getBiggerROI(mouse_selection_area, imp, 8);
				IJ.getImage().setRoi(roi);
				// focus on Trajectory (ROI)
				particleTracker3DModular.generateTrajFocusView(particleTracker3DModular.chosen_traj, roi);
			}
			else
			{
				// single-click - mark the selected trajectory by setting the ROI to the
				// trajectory mouse_selection_area
				Trajectory traj = particleTracker3DModular.iTrajectories.elementAt(particleTracker3DModular.chosen_traj);
				Roi roi = getBiggerROI(traj.trajectoryArea, imp, 2);
				this.imp.setRoi(roi);
			}
		}
		else
		{
			particleTracker3DModular.chosen_traj = -1;
			particleTracker3DModular.results_window.per_traj_label.setText("Trajectory (select from view)");

			if (e.getClickCount() == 2)
			{
				int					i							= imp.getCurrentSlice() - 1;
				int					j;
				int					x, y, dis;
				int					min_dis						= Integer.MAX_VALUE;
				int					chosen_particle				= -1;
				Vector	<Particle>	particles					= particleTracker3DModular.iFrames[i].getParticles();

				for(j = 0; j < particles.size(); j++)
				{
					dis =	((int) particles.elementAt(j).getX() - offscreenX) *
							((int) particles.elementAt(j).getX() - offscreenX) +
							((int) particles.elementAt(j).getY() - offscreenY) *
							((int) particles.elementAt(j).getY() - offscreenY);
						// if the distance for this particle  is lower than the min distance found
						// for all particles until now - save this particle for now
					if (dis < min_dis)
					{
						min_dis = dis;
						chosen_particle = j;
					}
				}

				particleTracker3DModular.results_window.text_panel.selectAll();
				particleTracker3DModular.results_window.text_panel.clearSelection();
				particleTracker3DModular.results_window.text_panel.appendLine("Mouse                                                      X position of " + Integer.toString(offscreenX) + " and Y position of " + Integer.toString(offscreenY) + "  on slice " + (i + 1) + "\n");

				if( min_dis < 10)
				{
					particleTracker3DModular.results_window.text_panel.appendLine ("Correspond to the particle " + Integer.toString(chosen_particle) + " with X position of " + Integer.toString((int) particles.elementAt(chosen_particle).getX()) + " and Y position of " + Integer.toString((int) particles.elementAt(chosen_particle).getY()) + "  on slice " + (i + 1) + "\n");
					particleTracker3DModular.results_window.per_traj_label.setText("Particle " + chosen_particle);

						min_dis		= Integer.MAX_VALUE;
					int	chosen_traj	= -1;

					while (iter.hasPrevious())
					{
						final Trajectory curr_traj = (Trajectory) iter.previous();
 
						dis =	((int) curr_traj.iParticles[i].getX() - (int) particles.elementAt(chosen_particle).getX()) *
								((int) curr_traj.iParticles[i].getX() - (int) particles.elementAt(chosen_particle).getX()) +
								((int) curr_traj.iParticles[i].getY() - (int) particles.elementAt(chosen_particle).getY()) *
								((int) curr_traj.iParticles[i].getY() - (int) particles.elementAt(chosen_particle).getY());
						// if the distance for this particle  is lower than the min distance found
						// for all trajectories until now - save this trajectory for now
								if(min_dis == 0)
						{
							chosen_traj = curr_traj.iSerialNumber + 1;
							break;
						}
						else	if (dis < min_dis)
							min_dis = dis;
					}

					if(min_dis == 0)
						particleTracker3DModular.results_window.text_panel.appendLine("This particle has already been chosen within the Trajectory " + Integer.toString(chosen_traj) + "\n");
					else
					{
						user_traj_particles_array[i] = particles.elementAt(chosen_particle);
						particleTracker3DModular.results_window.text_panel.appendLine("Particle added for new Trajectory\n");
					}
				}
//				else if (e.isShiftDown())
				else
				{
					// pointLocationsRefinement code use
					int		m, k, l, tx, ty;
					int		explore_radius		= 5;
					int		explore_mask_width	= 2 * explore_radius + 1;
					float	c;
					float	m0					= 0.0F;
					float	m2					= 0.0F;
					float	epsx				= 1.0F;
					float	epsy				= 1.0F;
					float	xf					= offscreenX;
					float	yf					= offscreenY;

					while (epsx > 0.5 || epsx < -0.5 || epsy > 0.5 || epsy < -0.5)
					{
						for(k = -explore_radius; k <= explore_radius; k++)
						{
							if((offscreenX + k) < 0 || (offscreenX + k) >= ic.getImage().getWidth())
								continue;
							x = offscreenX + k;

							for(l = -explore_radius; l <= explore_radius; l++)
							{
								if((offscreenY + l) < 0 || (offscreenY + l) >= ic.getImage().getHeight())
									continue;
								y = offscreenY + l;

								c = ic.getImage().getProcessor().getPixelValue(x, y) * (float) calculateMask(k, l, explore_mask_width);
								m0		+= c;
								epsx	+= (float) k * c;
								epsy	+= (float) l * c;
								m2		+= (float)(k * k + l * l) * c;
							}
						}
						epsx /= m0;
						epsy /= m0;
						m2	 /= m0;

						// This is a little hack to avoid numerical inaccuracy
						tx = (int)(10.0 * epsx);
						ty = (int)(10.0 * epsy);

							if((float)(tx) / 10.0 > 0.5)
						{
							if((int)xf + 1 < ic.getImage().getHeight())
								xf++;
						}
						else if((float)(tx) / 10.0 < -0.5)
						{
							if((int)xf - 1 >= 0)
								xf--;
						}
							if((float)(ty) / 10.0 > 0.5)
						{
							if((int)yf + 1 < ic.getImage().getWidth())
								yf++;
						}
						else if((float)(ty) / 10.0 < -0.5)
						{
							if((int)yf - 1 >= 0)
								yf--;
						}

						if((float)(tx) / 10.0 <= 0.5 && (float)(tx) / 10.0 >= -0.5 && (float)(ty) / 10.0 <= 0.5 && (float)(ty) / 10.0 >= -0.5)
							break;
					}
					xf += epsx;
					yf += epsy;

					particleTracker3DModular.results_window.text_panel.appendLine("corresponds to the optimized           X position of " + Float.toString(xf) + " and Y position of " + Float.toString(yf) + "  on slice " + (i + 1) + "\n");

					// Addition of this newly found particle to the corresponding particleTracker3DModular.iFrames[i].iParticles vector
					particleTracker3DModular.iFrames[i].addParticle(new Particle(xf, yf, 0.0f, i) );

					// Addition od this newly found particle to a new trajectory to be created
					user_traj_particles_array[i] = particles.elementAt(particles.size() - 1);
					particleTracker3DModular.results_window.text_panel.appendLine("New particle created with these positions and added to a new Trajectory\n");
				}
//				else
//					particleTracker3DModular.results_window.text_panel.appendLine("No corresponding particle found!!!\n");

				boolean trajectories_completed = true;
				for(j = 0; j < user_traj_particles_array.length; j++)
					if(user_traj_particles_array[j] == null)
						trajectories_completed = false;

				if(trajectories_completed)
				{
					// !!!! Go through a vector so that there is no link between user_traj_particles_array and all_traj once I set user_traj_particles_array to null
					Vector <Particle> curr_traj_particles = new Vector <Particle> ();
					for(j = 0; j < user_traj_particles_array.length; j++)
						curr_traj_particles.add(user_traj_particles_array[j]);

//					particleTracker3DModular.addTrajectory(curr_traj_particles.toArray(new Particle[0]));

					// Create the current trajectory
					Particle[] curr_traj_particles_array = new Particle[curr_traj_particles.size()];		// is equal to 2 in our case since we have only a link range of 1
					for(j = 0; j < curr_traj_particles.size(); j++)
						curr_traj_particles_array[j] = curr_traj_particles.get(j);
					particleTracker3DModular.addTrajectory(curr_traj_particles_array);

					particleTracker3DModular.results_window.text_panel.appendLine("Trajectory " + particleTracker3DModular.iTrajectories.size() + " added!");
//					particleTracker3DModular.results_window.text_panel.appendLine("Found " + number_of_trajectories + " Trajectories");

					IJ.getImage().killRoi();
					particleTracker3DModular.iTrajImg		= particleTracker3DModular.createHyperStackFromFrames();
					particleTracker3DModular.updateView		(IJ.getImage(), particleTracker3DModular.iTrajImg);

					Arrays.fill(user_traj_particles_array, null);
				}
			}
		}

		// ?????????????? is this the only way to update
//		this.showSlice(this.imp.getCurrentSlice() + 1);
//		this.showSlice(this.imp.getCurrentSlice() - 1);
	}

	@Override
	public void mouseClicked(MouseEvent e) { }

	@Override
	public void mouseEntered(MouseEvent arg0) { }

	@Override
	public void mouseExited(MouseEvent arg0) { }

	@Override
	public void mouseReleased(MouseEvent arg0) { }

	// (non-Javadoc) @see java.awt.event.KeyListener#keyPressed(java.awt.event.keyEvent)
	public void keyPressed(KeyEvent e)
	{
		int keyCode = e.getKeyCode();
		int flags   = e.getModifiers();
//		e.consume();

		if (keyCode == KeyEvent.VK_DELETE && flags == InputEvent.CTRL_MASK)
		{
			if (particleTracker3DModular.chosen_traj == -1 && IJ.getImage().getRoi() == null)
				IJ.error("Please make a selection\n" +
						 "or\n" +
						 "choose a trajectory first\n\n" +
						 "Click and drag the mouse on the active image to make a selection in 'All trajectories' display\n" +
						 "or\n" +
						 "Click with the mouse on a trajectory in 'All trajectories' display to select one");
			else
			{
				if (particleTracker3DModular.chosen_traj != -1)
				{
					particleTracker3DModular.iTrajectories.remove(particleTracker3DModular.chosen_traj);

					IJ.getImage().killRoi();
					particleTracker3DModular.iTrajImg		= particleTracker3DModular.createHyperStackFromFrames();
					particleTracker3DModular.updateView		(IJ.getImage(), particleTracker3DModular.iTrajImg);

					particleTracker3DModular.results_window.text_panel.selectAll();
					particleTracker3DModular.results_window.text_panel.clearSelection();
					particleTracker3DModular.results_window.text_panel.appendLine	("Trajectory " + (particleTracker3DModular.chosen_traj + 1) + " deleted!");
					particleTracker3DModular.results_window.text_panel.appendLine	("Found " + particleTracker3DModular.iTrajectories.size() + " Trajectories");

					particleTracker3DModular.chosen_traj = -1;
					particleTracker3DModular.results_window.per_traj_label.setText("Trajectory (select from view)");
				}
				else
				{
					Roi user_roi = IJ.getImage().getRoi();
					int deleted = 0;
					IJ.getImage().setOverlay(null);
					IJ.getImage().killRoi();
					particleTracker3DModular.results_window.text_panel.selectAll();
					particleTracker3DModular.results_window.text_panel.clearSelection();
					// iterate of all trajectories
					for (int j = particleTracker3DModular.iTrajectories.size() - 1; j > 0; j--)
					{
						for (int i = 0; i < particleTracker3DModular.iTrajectories.elementAt(j).iParticles.length; i++)
						{
							// if a particle in the trajectory is within the ROI
							// print traj information to screen and go to next trajectory
							if (user_roi.getBounds().contains(particleTracker3DModular.iTrajectories.elementAt(j).iParticles[i].getX(), particleTracker3DModular.iTrajectories.elementAt(j).iParticles[i].getY()) && particleTracker3DModular.iTrajectories.elementAt(j).to_display)
							{
								particleTracker3DModular.results_window.text_panel.appendLine("Trajectory " + j + " deleted!");
								particleTracker3DModular.iTrajectories.remove(j);
								deleted++;
								break;
							}
						} // for
					} // while
					particleTracker3DModular.iTrajImg		= particleTracker3DModular.createHyperStackFromFrames();
					particleTracker3DModular.updateView		(IJ.getImage(), particleTracker3DModular.iTrajImg);
					particleTracker3DModular.results_window.text_panel.appendLine("\n" + deleted + " Trajectories have been deleted!\n");
					particleTracker3DModular.results_window.text_panel.appendLine("Found " + particleTracker3DModular.iTrajectories.size() + " Trajectories");
				}
			}
		}
	}

	public void keyReleased(KeyEvent e) { }

	public void keyTyped(KeyEvent e) { }
}