import ij.IJ;
import ij.ImagePlus;
import ij.gui.Arrow;
import ij.gui.GenericDialog;
import ij.plugin.LutLoader;
import ij.plugin.PlugIn;
import ij.process.ByteProcessor;
import ij.process.ColorProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import ij.io.OpenDialog;
import java.awt.Color;
import java.awt.Font;
import java.awt.image.IndexColorModel;
import java.io.BufferedReader;
import java.io.FileReader;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Locale;


/*
This is the code for plotting the PIV result. 
For more detail please see:
https://sites.google.com/site/qingzongtseng/piv
or contact me. 
By TSENG Qingzong (qztseng /at/ gmail.com)
*/

public class PIV_plot implements PlugIn
{
	private		static boolean					plugin					= false;
	private		static boolean					autoScale;
	private		static boolean					plotPaScalebar;
	private		static boolean					plotPaVector;
	private		static boolean					plotPaMagnitude;
	private		static boolean					plotPaMagnitudeVector;
	private		static boolean					plotPaMagnitudeX;
	private		static boolean					plotPaMagnitudeY;
	private		static int						pictureWidth			= 0;
	private		static int						pictureHeight			= 0;
	private		static int						insetWidth, insetHeight;
	private		static int						maxLenPixel				= 24;		// The longest arrow size in pixel in autoscaled plot
	private		static int						lenPixel;					// The longest arrow size in pixel in non-autoscaled plot
	private		static double					scale					= 0D;
	public 		static double					colorMax				= 600D;
	private		static double					max;
	private		static String					lut;
	private		static IndexColorModel			icm						= null;
	private		static int				[]		dimensions;					// record data dimensions as array (nColumn, nRow, vectorSpace)
	private		static byte				[]		lutR, lutG, lutB;
	private		static String			[]		pathname;
	protected	static String			[]		multitxt				= {"  1   1   1", "  1   1   1", "  1   1   2", "  1   1   5", "  1   1   7", "  1   1  11", "  1   1  16", "  1   1  20", "  1   1  26", "  1   1  32", "  1   1  38", "  1   1  45", "  1   1  53", "  1   1  60", "  1   1  67", "  1   1  75", "  1   1  83", "  1   1  91", "  1   1  99", "  1   1 107", "  1   1 114", "  1   1 122", "  1   1 130", "  1   1 137", "  1   1 144", "  1   1 150", "  1   1 156", "  1   1 162", "  1   1 168", "  1   1 174", "  1   1 180", "  1   1 186", "  1   1 191", "  1   1 197", "  1   1 203", "  1   1 209", "  1   1 215", "  1   1 221", "  1   1 227", "  1   1 233", "  1   1 239", "  1   1 245", "  1   1 251", "  1   4 254", "  1  10 254", "  1  16 254", "  1  22 254", "  1  28 254", "  1  34 254", "  1  40 254", "  1  46 254", "  1  52 254", "  1  58 254", "  1  64 254", "  1  69 254", "  1  75 254", "  1  81 254", "  1  87 254", "  1  93 254", "  1  99 254", "  1 105 254", "  1 111 254", "  1 117 254", "  1 123 254", "  1 129 254", "  1 135 254", "  1 141 254", "  1 147 254", "  1 153 254", "  1 159 254", "  1 165 254", "  1 171 254", "  1 177 254", "  1 183 254", "  1 189 254", "  1 194 254", "  1 200 254", "  1 206 254", "  1 212 254", "  1 218 254", "  1 224 254", "  1 230 254", "  1 236 254", "  1 242 254", "  1 248 254", "  1 254 254", "  1 254 248", "  1 254 242", "  1 254 236", "  1 254 230", "  1 254 224", "  1 254 218", "  1 254 212", "  1 254 206", "  1 254 200", "  1 254 194", "  1 254 189", "  1 254 183", "  1 254 177", "  1 254 171", "  1 254 165", "  1 254 159", "  1 254 153", "  1 254 147", "  1 254 141", "  1 254 135", "  1 254 129", "  1 254 123", "  1 254 117", "  1 254 111", "  1 254 105", "  1 254  99", "  1 254  93", "  1 254  87", "  1 254  81", "  1 254  75", "  1 254  69", "  1 254  64", "  1 254  58", "  1 254  52", "  1 254  46", "  1 254  40", "  1 254  34", "  1 254  28", "  1 254 22", "  1 254   16", "  1 254  10", "  1 254   4", "  4 254   1", " 10 254   1", " 16 254   1", " 22 254   1", " 28 254   1", " 34 254   1", " 40 254   1", " 46 254   1", " 52 254   1", " 58 254   1", " 64 254   1", " 69 254   1", " 75 254   1", " 81 254   1", " 87 254   1", " 93 254   1", " 99 254   1", "105 254   1", "111 254   1", "117 254   1", "123 254   1", "129 254   1", "135 254   1", "141 254   1", "147 254   1", "153 254   1", "159 254   1", "165 254   1", "171 254   1", "177 254   1", "183 254   1", "189 254   1", "194 254   1", "200 254   1", "206 254   1", "212 254   1", "218 254   1", "224 254   1", "230 254   1", "236 254   1", "242 254   1", "248 254   1", "254 254   1", "254 248   1", "254 242   1", "254 236   1", "254 230   1", "254 224   1", "254 218   1", "254 212   1", "254 206   1", "254 200   1", "254 194   1", "254 189   1", "254 183   1", "254 177   1", "254 171   1", "254 165   1", "254 159   1", "254 153   1", "254 147   1", "254 141   1", "254 135   1", "254 129   1", "254 123   1", "254 117   1", "254 111   1", "254 105   1", "254 99    1", "254  93   1", "254  87   1", "254  81   1", "254  75   1", "254  69   1", "254  64   1", "254  58   1", "254  52   1", "254  46   1", "254  40   1", "254  34   1", "254  28   1", "254  22   1", "254  16   1", "254  10   1", "254   4   1", "254   4   4", "254  10  10", "254  16  16", "254  22  22", "254  28  28", "254  34  34", "254  40  40", "254  46  46", "254  52  52", "254  58  58", "254  64  64", "254  69  69", "254  75  75", "254  81  81", "254  87  87", "254  93  93", "254  99  99", "254 105 105", "254 111 111", "254 118 118", "254 125 125", "254 133 133", "254 141 141", "254 148 148", "254 156 156", "254 164 164", "254 172 172", "254 180 180", "254 188 188", "254 195 195", "254 202 202", "254 210 210", "254 217 217", "254 223 223", "254 229 229", "254 235 235", "254 239 239", "254 244 244", "254 248 248", "254 250 250", "254 253 253", "254 254 254", "254 254 254"};
	protected	static String			[]		spet					= {"  4   1   4", "  4   1   4", "  7   1   8", "  9   1  13", " 10   1  17", " 13   1  21", " 16   1  25", " 18   1  30", " 19   1  34", " 22   1  38", " 25   1  42", " 27   1  47", " 28   1  51", " 31   1  55", " 34   1  59", " 35   1  64", " 38   1  67", " 41   1  71", " 43   1  75", " 44   1  80", " 47   1  84", " 50   1  88", " 52   1  92", " 53   1  97", " 56   1 101", " 59   1 105", " 61   1 109", " 62   1 114", " 64   1 118", " 67   1 122", " 68   1 127", " 68   1 127", " 67   1 131", " 64   1 135", " 62   1 139", " 61   1 143", " 59   1 147", " 56   1 151", " 53   1 155", " 52   1 159", " 50   1 163", " 47   1 167", " 44   1 171", " 43   1 175", " 41   1 179", " 38   1 183", " 35   1 187", " 34   1 191", " 31   1 194", " 28   1 198", " 27   1 202", " 25   1 206", " 22   1 210", " 19   1 214", " 18   1 218", " 16   1 222", " 13   1 226", " 10   1 230", "  9   1 234", "  7   1 238", "  4   1 242", "  2   1 246", "  1   1 254", "  1   1 254", "  1   5 246", "  1  10 242", "  1  14 238", "  1  19 234", "  1  23 230", "  1  28 226", "  1  32 222", "  1  37 218", "  1  41 214", "  1  46 210", "  1  50 206", "  1  55 202", "  1  59 198", "  1  64 194", "  1  67 191", "  1  72 187", "  1  76 183", "  1  81 179", "  1  85 175", "  1  90 171", "  1  94 167", "  1  99 163", "  1 103 159", "  1 108 155", "  1 112 151", "  1 117 147", "  1 121 143", "  1 126 139", "  1 130 135", "  1 135 131", "  1 139 127", "  1 144 122", "  1 148 118", "  1 153 114", "  1 157 109", "  1 162 105", "  1 166 101", "  1 171  97", "  1 175  92", "  1 180  88", "  1 184  84", "  1 189  80", "  1 192  75", "  1 203  71", "  1 203  67", " 10 205  64", " 19 207  59", " 28 209  55", " 37 211  51", " 46 213  47", " 55 215  42", " 64 217  38", " 72 219  34", " 81 221  30", " 90 223  25", " 99 225  21", "108 227  17", "117 229  13", "126 231  11", "135 233   8", "144 235   5", "153 237   2", "162 239   1", "171 241   1", "180 242   1", "189 244   1", "197 245   1", "206 247   1", "215 248   1", "224 250   1", "233 251   1", "242 253   1", "254 254   1", "254 254   1", "254 253   1", "254 251   1", "254 249   1", "254 247   1", "254 245   1", "254 243   1", "254 241   1", "254 239   1", "254 237   1", "254 235   1", "254 233   1", "254 231   1", "254 229   1", "254 227   1", "254 225   1", "254 223   1", "254 221   1", "254 219   1", "254 217   1", "254 215   1", "254 213   1", "254 211   1", "254 209   1", "254 207   1", "254 205   1", "254 203   1", "254 201   1", "254 199   1", "254 197   1", "254 195   1", "254 193   1", "254 191   1", "254 190   1", "254 188   1", "254 186   1", "254 184   1", "254 182   1", "254 180   1", "254 178   1", "254 176   1", "254 174   1", "254 172   1", "254 170   1", "254 168   1", "254 166   1", "254 164   1", "254 162   1", "254 160   1", "254 158   1", "254 156   1", "254 154   1", "254 152   1", "254 150   1", "254 148   1", "254 146   1", "254 144   1", "254 142   1", "254 140   1", "254 138   1", "254 136   1", "254 134   1", "254 132   1", "254 130   1", "254 128   1", "254 126   1", "254 124   1", "254 122   1", "254 120   1", "254 118   1", "254 116   1", "254 114   1", "254 112   1", "254 110   1", "254 108   1", "254 106   1", "254 104   1", "254 102   1", "254 100   1", "254  98   1", "254  96   1", "254  94   1", "254  92   1", "254  90   1", "254  88   1", "254  86   1", "254  84   1", "254  82   1", "254  80   1", "254  78   1", "254  76   1", "254  74   1", "254  72   1", "254  6  9 1", "254  66   1", "254  64   1", "254  62   1", "254  59   1", "254  56   1", "254  54   1", "254  51   1", "254  48   1", "254  45   1", "254  43   1", "254  40   1", "254  37   1", "254  34   1", "254  32   1", "254  29   1", "254  26   1", "254  23   1", "254  21   1", "254  18   1", "254  15   1", "254  12   1", "254  10   1", "254   7   1", "254   4   1", "254   4   1"};
	protected	static String			[]		white					= {"255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255", "255 255 255"};
	protected	static String			[]		colors					= null;
	private		static double			[][]	data;
	private		static double			[][]	magArr;
	private		static double			[][]	XArr;
	private		static double			[][]	YArr;

	public void run(String arg)
	{
		if(IJ.versionLessThan("1.53d"))
			return;

		try 
		{
			pathname = getFilePath("Select the data for plotting");
		}
		catch (Exception e)
		{
			IJ.error(e.getMessage());
			return;
		}

		plotFTTCdata();

		return;
	}
	
	private static void plotFTTCdata()
	{
		try
		{
			data = loadMatrixFromFile(pathname[0] + pathname[1], 0);
		}
		catch (Exception e)
		{
			IJ.error(e.getMessage());
		}

		// get the data dimensions and set the picture dimensions
		dimensions = getDimensions(data);
		// dimensions[0] is the number of data points in each row    (nx)
		// dimensions[1] is the number of data points in each column (ny)
		// dimensions[2] is the spacing between each data point      (in pixel)

		// calculate the magnitude(norm of the vector)
		magArr = get2DElement(data, dimensions[0], dimensions[1], 4);

		if (!plugin)
			if (!getParams())
				return;

		// Calculate the inset width and height, as well as the picture width and height
		insetWidth	= (dimensions[0] - 1) * dimensions[2];
		insetHeight	= (dimensions[1] - 1) * dimensions[2];
		if (pictureHeight == 0 && pictureWidth == 0)
		{
			pictureWidth  = dimensions[0] * dimensions[2] + (int) (2 * data[0][0]);
			pictureHeight = dimensions[1] * dimensions[2] + (int) (2 * data[0][1]);
		}
		
		// load the lut
		loadLut(lut);

		if (autoScale || plotPaScalebar)
		{
			max = findMax2DArray(magArr);
			if (autoScale)
			{
				scale    = maxLenPixel / max;
				colorMax = max;
			}
		}

		plotData(pathname[1], pictureWidth, pictureHeight, insetWidth, insetHeight, scale, colors, icm, dimensions, data, magArr, plotPaScalebar, plotPaVector, plotPaMagnitude, plotPaMagnitudeVector, plotPaMagnitudeX, plotPaMagnitudeY);
	}
	
	public static void plotData(String fileName, int pictureWidth, int pictureHeight, int insetWidth, int insetHeight, double plotScale, String[] plotColors, IndexColorModel plotColorModel, int[] plotDataDimensions, double[][] plotDataMatrix, double[][] plotDataMagnitude, boolean plotPaScalebar, boolean plotPaVector, boolean plotPaMagnitude, boolean plotPaMagnitudeVector, boolean plotPaMagnitudeX, boolean plotPaMagnitudeY)
	{
		ImageProcessor	ip, ip2;
		ImagePlus		imp;

		// draw the plotPaScalebar
		if (plotPaScalebar)
			makeScaleGraph(plotScale, fileName);

		// draw the vector plot
		if (plotPaVector)
		{
			ip = new ColorProcessor(pictureWidth, pictureHeight);
			drawVectors(ip, plotDataDimensions, plotDataMatrix, plotDataMagnitude, plotScale, plotColors);
			imp = new ImagePlus("Traction_Vector_Plot_" + fileName, ip);
			imp.show();
		}

		// draw the magnitude plot
		if (plotPaMagnitude)
		{
			ip2 = new FloatProcessor(doubleArrayToFloat(plotDataMagnitude));
			ip2	.setInterpolationMethod(ImageProcessor.BICUBIC);
			ip2 = (FloatProcessor) ip2.resize(insetWidth, insetHeight);
			ip2	.setColorModel(plotColorModel);
			ip	= ip2.createProcessor(pictureWidth, pictureHeight);
//			ip	.setValue(0.0);
//			ip	.fill();
			ip	.insert(ip2, (int) plotDataMatrix[0][0], (int) plotDataMatrix[0][1]);
			imp	= new ImagePlus("Traction_Magnitude_Map_" + fileName, ip);
			imp	.show();	
		}

		// draw the magnitude and vector plot
		if (plotPaMagnitudeVector)
		{
			ip2 = new FloatProcessor(doubleArrayToFloat(plotDataMagnitude));
			ip2	.setInterpolationMethod(ImageProcessor.BICUBIC);
			ip2 = (FloatProcessor) ip2.resize(insetWidth, insetHeight);
			ip2	.setColorModel(plotColorModel);
			ip	= ip2.createProcessor(pictureWidth, pictureHeight);
			ip	.insert(ip2, (int) plotDataMatrix[0][0], (int) plotDataMatrix[0][1]);
			drawVectors(ip, plotDataDimensions, plotDataMatrix, plotDataMagnitude, plotScale, white);
			imp	= new ImagePlus("Traction_Magnitude_Map_and_Vector_Plot_" + fileName, ip);
			imp	.show();
		}

		// draw the X plot
		if (plotPaMagnitudeX)
		{
			XArr = get2DElement(plotDataMatrix, plotDataDimensions[0], plotDataDimensions[1], 2);
			ip2 = new FloatProcessor(doubleArrayToFloat(XArr));
			ip2	.setInterpolationMethod(ImageProcessor.BICUBIC);
			ip2 = (FloatProcessor) ip2.resize(insetWidth, insetHeight);
			ip2	.setColorModel(plotColorModel);
			ip	= ip2.createProcessor(pictureWidth, pictureHeight);
			ip	.insert(ip2, (int) plotDataMatrix[0][0], (int) plotDataMatrix[0][1]);
			imp	= new ImagePlus("Traction_X_Map_" + fileName, ip);
			imp	.show();
		}

		// draw the Y plot
		if (plotPaMagnitudeY)
		{
			YArr = get2DElement(plotDataMatrix, plotDataDimensions[0], plotDataDimensions[1], 3);
			ip2 = new FloatProcessor(doubleArrayToFloat(YArr));
			ip2	.setInterpolationMethod(ImageProcessor.BICUBIC);
			ip2 = (FloatProcessor) ip2.resize(insetWidth, insetHeight);
			ip2	.setColorModel(plotColorModel);
			ip	= ip2.createProcessor(pictureWidth, pictureHeight);
			ip	.insert(ip2, (int) plotDataMatrix[0][0], (int) plotDataMatrix[0][1]);
			imp	= new ImagePlus("Traction_Y_Map_" + fileName, ip);
			imp	.show();
		}
	}

	public static void drawPlot(String _pathName, String _fileName, boolean _autoScale, int _lenPixel, double _colorMax, int _pictureWidth, int _pictureHeight, boolean _plotPaScalebar, boolean _plotPaVector, boolean _plotPaMagnitude, boolean _plotPaMagnitudeVector, boolean _plotPaMagnitudeX, boolean _plotPaMagnitudeY, String _lut)
	{
		plugin					= true;
		pathname				= new String[2];
		pathname[0]				= _pathName;
		pathname[1]				= _fileName;
		autoScale				= _autoScale;
 		lenPixel				= _lenPixel;
		colorMax				= _colorMax;
		scale					= (double)	lenPixel / colorMax;
		pictureWidth			= _pictureWidth;
		pictureHeight			= _pictureHeight;
		plotPaScalebar			= _plotPaScalebar;
		plotPaVector			= _plotPaVector;
		plotPaMagnitude			= _plotPaMagnitude;
		plotPaMagnitudeVector	= _plotPaMagnitudeVector;
		plotPaMagnitudeX		= _plotPaMagnitudeX;
		plotPaMagnitudeY		= _plotPaMagnitudeY;
		lut						= _lut;

		plotFTTCdata();
	}

	protected static void makeScaleGraph(double sc, String pathname)
	{
		ImagePlus impS;
		ColorProcessor cpScale;
		int nScaleArrows = 5;
		int colorBarHeight = 500;
		int colorBarWidth = 30;
		int sOffX = 10;
		int sOffY = 20;
		int scaleGraphWidth;
		scaleGraphWidth = (int) (colorMax * sc) + sOffX * 2;

		cpScale = makeVerticalColorBar(colorBarWidth, colorBarHeight, scaleGraphWidth, sOffY);
		addScaleArrows(cpScale, nScaleArrows, colorBarHeight, sc, sOffX, sOffY);
		impS = new ImagePlus("Scale_Graph_" + pathname, cpScale);
		impS.show();

		return;
	}

	private static void addScaleArrows(ColorProcessor cp, int nArrows, int bHeight, double vs, int sOffX, int sOffY)
	{
		int				i;
		int				xLabel, yLabel;
		double	[][]	sArr			= new double[nArrows + 1][4];
		double	[][]	sMagArr			= new double[1][nArrows + 1];
		int		[]		dim				= {1, nArrows, bHeight / nArrows};
		Color			c;
		Font			font;

		for (i = 0; i <= nArrows; i++)
		{
			sArr[i][0] = sOffX;
			sArr[i][1] = sOffY + (bHeight / nArrows) * i;
			sArr[i][3] = 0;
			sArr[i][2] = colorMax * ((nArrows - i) / (double) nArrows);
			sMagArr[0][i] = sArr[i][2];
		}
		drawVectors(cp, dim, sArr, sMagArr, vs, colors);

		c = Color.white;
		cp.setColor(c);
		font = new Font("SansSerif", Font.BOLD, 12);
		cp.setFont(font);
		cp.setAntialiasedText(true);

		for (i = 0; i <= nArrows; i++)
		{
			xLabel = sOffX;
			yLabel = (int) sArr[i][1] - 3;
			String label;
			label = IJ.d2s(colorMax * ((nArrows - i) / (double) nArrows), 0);
			cp.drawString(label, xLabel, yLabel);
		}
	}

	private static ColorProcessor makeVerticalColorBar(int width, int height, int offX, int offY)
	{	// create a 0-255 vertical ramp image. Modified from the ImageJ newImage.class
		int					i, j;
		byte			[]	pixels	= new byte[width * height];
		ImageProcessor		ip1, ip2;
		ColorProcessor		cp;

		for (j = 0; j < height; j++)
			for (i = 0; i < width; i++)
				pixels[j * width + i] = (byte) (((height - 1 - j) * 256) / height);

		ip1	= new ByteProcessor(width, height, pixels, icm);
		ip2	= ip1.createProcessor(width + offX, height + offY);
		ip2	.insert(ip1, offX, offY);
		cp	= ip2.convertToColorProcessor();

		return cp;
	}

	public static String[] getFilePath(String str) throws Exception
	{
		OpenDialog od = new OpenDialog(str, "");
		if (od.getDirectory() == null || od.getFileName() == null)
			throw new Exception("No file selected");

		String[] pn = {od.getDirectory(), od.getFileName()};

		return pn;
	}

/*
	double[][] readArrayFromFile(String pathname) throws IOException
	{
		int numOfRows = 0;
		int i = 0;
		int numOfCol = 0;
		int j = 0;
		double[] d = new double[25];
		// array list for conveniently appending the parsed rows
		// do not use a LinkedList here: bad performance
		ArrayList al = new ArrayList();
		// count lines that contain Tecplot key words
		BufferedReader br = new BufferedReader(new FileReader(pathname));

		// configuring StreamTokenizer
		StreamTokenizer st = new StreamTokenizer(br);
		st.eolIsSignificant(true);
		st.resetSyntax();
		st.wordChars('0', '9');
		st.wordChars('-', '-');
		st.wordChars('+', '+');
		st.wordChars('e', 'e');
		st.wordChars('E', 'E');
		st.wordChars('.', '.');
		st.whitespaceChars(' ', ' ');
		st.whitespaceChars('\t', '\t');
		int type = -1;
		while ((type = st.nextToken()) != StreamTokenizer.TT_EOF)
		{
			switch (type)
			{
				case StreamTokenizer.TT_WORD:
					d[j] = Double.parseDouble(st.sval);
					j++;
					break;
				case StreamTokenizer.TT_EOL:
					// at the end of the line, the data is appended at the ArrayList
					// use clone() to copy the object and not just its reference
					al.add(data.clone());
					numOfRows++;
					numOfCol = j;
					j = 0;
					break;
				default:
					break;
			}
		}
		br.close();
		// copying the data from the ArrayList into a double-array
		double[][] array = new double[numOfRows][numOfCol];
		for (i = 0; i < numOfRows; ++i)
		{
			d = (double[]) al.get(i);
			System.arraycopy(data, 0, array[i], 0, numOfCol);
		}
		al.clear();
		return (array);
	}
*/

	private static boolean getParams()
	{
		// generate the items of the lutText array
		int					i, lutIndex	= 0;
		String	[]			lutList, lutText;
		lutList				= IJ.getLuts();
		lutText				= new String[lutList.length + 2];
		lutText[0]			= "multitxt";
		lutText[1]			= "S_Pet";
		for(i = 0; i < lutList.length; i++)
			lutText[i + 2]	= lutList[i];

		GenericDialog gd = new GenericDialog("PIV plot");
		gd.addMessage		("Vector plot parameters:");
		gd.addCheckbox		("Autoscale_vector_plot?"									, false);
		gd.addMessage		("Set the following two scale factors if autoscale was not set:");
//		gd.addNumericField	("Max_vector_length"										, 25, 0);
		gd.addNumericField	("Max_vector_length"										, maxLenPixel, 0);
//		gd.addNumericField	("Max_vector_value "										, 500, 1);
		gd.addNumericField	("Max_vector_value "										, findMax2DArray(magArr), 3);
		gd.addMessage		("___________________________________________");
		gd.addMessage		("");
//		gd.addNumericField	("Plot_width (pixel): "										, 0, 0);
		gd.addNumericField	("Plot_width (pixel): "										, 1200, 0);
//		gd.addNumericField	("Plot_height (pixel):"										, 0, 0);
		gd.addNumericField	("Plot_height (pixel):"										, 1200, 0);
		gd.addMessage		("The plot dimensions will be approximated from the data\n        if both the width and height values are set to 0");
		gd.addMessage		("___________________________________________");
		gd.addMessage		("");
		gd.addCheckbox		("Show_scale_graph?"										, true);
		gd.addCheckbox		("Draw_vector_plot?"										, true);
		gd.addCheckbox		("Draw_magnitude_plot?"										, true);
		gd.addCheckbox		("Draw_magnitude_and_vector_plots_together?"				, true);
		gd.addMessage		("___________________________________________");
		gd.addMessage		("");
		gd.addCheckbox		("Draw_X_magnitude_plot?"									, false);
		gd.addCheckbox		("Draw_Y_magnitude_plot?"									, false);
		gd.addMessage		("___________________________________________");
		gd.addMessage		("");
		gd.addChoice		("                LUT for color coding"						, lutText, lutText[lutIndex]);
		gd.showDialog();

		autoScale				=			gd.getNextBoolean		();
 		lenPixel				= (int)		gd.getNextNumber		();
//		colorMax				= (double)	gd.getNextNumber		();
		colorMax				=			gd.getNextNumber		();
		scale					= (double)	lenPixel / colorMax;
		pictureWidth			= (int)		gd.getNextNumber		();
		pictureHeight			= (int)		gd.getNextNumber		();
		plotPaScalebar			=			gd.getNextBoolean		();
		plotPaVector			=			gd.getNextBoolean		();
		plotPaMagnitude			=			gd.getNextBoolean		();
		plotPaMagnitudeVector	=			gd.getNextBoolean		();
		plotPaMagnitudeX		=			gd.getNextBoolean		();
		plotPaMagnitudeY		=			gd.getNextBoolean		();
		lutIndex				=			gd.getNextChoiceIndex	();
		lut						=			lutText[lutIndex];

		if (gd.wasCanceled())
			return false;

		return true;
	}

	protected static int[] getDimensions(double[][] array)
	{
		int		i;												// dm[0] is the number of data points in each row    (nx)
		int	[]	dm = new int[3];								// dm[1] is the number of data points in each column (ny)
																// dm[2] is the spacing between each data point      (in pixel)
		if ((array[1][0] - array[0][0]) == 0)
		{
			dm[2] = (int) (array[1][1] - array[0][1]);
			for (i = 1; i < array.length; i++)
				if (array[i][0] == array[i - 1][0])
					dm[1]++;
				else
					break;

			dm[1]++;
			dm[0] = array.length / dm[1];
		}
		else
		{
			dm[2] = (int) (array[1][0] - array[0][0]);
			for (i = 1; i < array.length; i++)
				if (array[i][1] == array[i - 1][1])
					dm[0]++;
				else
					break;

			dm[0]++;
			dm[1] = array.length / dm[0];
		}
		return dm;
	}

	protected static void drawVectors(ImageProcessor ip, int[] dim, double[][] d, double[][] magArr, double vs, String[] lut)
	{
		int			i;
		int			cIndex;
		double		x0, y0;			// vector origin
		double		dx, dy;			// vector components
		double		l;				// vector length
		String	[]	rgb		= new String[3];
		Arrow		arrow;

		// draw vectors
		for (i = 0; i < d.length; ++i)
		{
			ip.setColor(new Color(255, 0, 0));
			x0 = d[i][0];
			y0 = d[i][1];
			dx = d[i][2] * vs;
			dy = d[i][3] * vs;

			l = magArr[i % dim[0]][(int) Math.floor(i / dim[0])];
//			l = Math.sqrt((dx * dx + dy * dy));
//			if (autoScale) {
//				cIndex = (int) (l * 255 / max);
//			} else {
//				cIndex = (int) (l * 255 / colorMax);
//			}
			cIndex = (int) (l * 255 / colorMax);
			if (cIndex < 0)
				cIndex = 0;

			if (cIndex > 255)
				cIndex = 255;

			rgb = lut[cIndex].trim().split("\\s+");
			ip.setColor(new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]), Integer.parseInt(rgb[2])));
			arrow = new ij.gui.Arrow(x0, y0, x0 + dx, y0 + dy);
			if (l * vs / 3.5 > 4)
				arrow.setHeadSize(l * vs / 4);
			else
				arrow.setHeadSize(4);

//			if(dx!=0 || dy!=0)
//			if((dx!=0 || dy!=0) && !Double.isNaN(dx) && !Double.isNaN(dy)) {
			if(Math.abs(dx) > 1.0e-13 || Math.abs(dy) > 1.0e-13)
				ip.draw(arrow);
		}
	}

	private static double[][] calcMag(double[][][] vectorArr)
	{
		int				i, j;
		double	[][]	Arr = new double[vectorArr.length][vectorArr[0].length];

		for (j = 0; j < vectorArr.length; j++)
			for (i = 0; i < vectorArr[0].length; i++)
				Arr[j][i] = Math.sqrt(vectorArr[j][i][2] * vectorArr[j][i][2] + vectorArr[j][i][3] * vectorArr[j][i][3]);

		return Arr;
	}

	protected static double[][][] convert2DPivTo3D(double[][] _piv, int nx, int ny)
	{
		boolean			fx;
		int				i, j, k;
		double	[][][]	newPIV = new double[nx][ny][_piv[0].length];

		if (_piv[0][0] == _piv[1][0])			// check the _piv array is fixing x first or y first
			fx = true;							// x coordinates is fixed first
		else
			fx = false;

		for (j = 0; j < ny; j++)
			for (i = 0; i < nx; i++)
				for (k = 0; k < _piv[0].length; k++)
					if (fx)
						newPIV[i][j][k] = _piv[ny * i + j][k];
					else
						newPIV[i][j][k] = _piv[nx * j + i][k];

		return newPIV;
	}

	protected static double findMax2DArray(double[][] M)
	{
		int		i, j;
		double	m = M[0][0];

		for (i = 0; i < M.length; i++)
			for (j = 0; j < M[0].length; j++)
				if (M[i][j] > m)
					m = M[i][j];

		return m;
	}

	static protected float[][] doubleArrayToFloat(double[][] dArr)
	{
		int				i, j;
		float	[][]	fArr = new float[dArr.length][dArr[0].length];

		for (i = 0; i < fArr.length; i++)
			for (j = 0; j < fArr[0].length; j++)
				fArr[i][j] = (float) dArr[i][j];

		return fArr;
	}

	private static void splitLUTtoRGB(String[] _lut)
	{
		int			i;
		String	[]	rgb	= new String[3];
		lutR			= new byte[_lut.length];
		lutG			= new byte[_lut.length];
		lutB			= new byte[_lut.length];

		for(i = 0; i < _lut.length; i++)
		{
			rgb		= _lut[i].trim().split("\\s+");
			lutR[i] = (byte) Integer.parseInt(rgb[0]);
			lutG[i] = (byte) Integer.parseInt(rgb[1]);
			lutB[i] = (byte) Integer.parseInt(rgb[2]);
		}

		return;
	}

	protected static void loadLut(String lut)
	{
		int i, size;
   
		if		(lut.equals("S_Pet"))
			colors = spet;
		else if (lut.equals("multitxt"))
			colors = multitxt;
		else
		{
			try
			{
				LutLoader lutLoader	= new LutLoader();
				icm					= lutLoader.getLut(lut);
				colors				= null;
			}
			catch (Exception e)
			{
				IJ.error("Error reading LUT file:\n" + IJ.getDirectory("luts") + lut.replace(" ", "_") + ".lut");
			}
		}
		if(colors == null)
		{
			size	= icm.getMapSize();
			colors	= new String[size];
			lutR	= new byte	[size];
			lutG	= new byte	[size];
			lutB	= new byte	[size];
			icm		.getReds	(lutR);
			icm		.getGreens	(lutG);
			icm		.getBlues	(lutB);
			for (i = 0; i < size; i++)
				colors[i] = pad(String.valueOf(lutR[i] & 255), 3) + pad(String.valueOf(lutG[i] & 255), 4) + pad(String.valueOf(lutB[i] & 255), 4);
		}
		else
		{
			splitLUTtoRGB(colors);
			icm = new IndexColorModel(8, 256, lutR, lutG, lutB);	
		}
	}

	public static String pad(String s, int digits)
	{
		String str = "" + s;
		while(str.length() < digits)
			str = " " + str;
		return str;
	}

	protected static IndexColorModel getCm()
	{
		return icm;
	}

	protected void calculateMag(double[][] _d)
	{
		double	[][][]	matrix;
		// convert the 2D ([#data points][x,y,dx,dy]) PIV array to 3D ([#data points in width][#data points in height][x,y,dx,dy])
		matrix = convert2DPivTo3D(_d, dimensions[0], dimensions[1]);
		magArr = calcMag(matrix);
	}

	protected static double[][] loadMatrixFromFile(String path, int col) throws Exception
	{
		int					i, j;
		String				line;
		double[][]			matrix;
		ArrayList<String[]> aL		= new ArrayList<String[]>(); 	// an array list to hold all vector info as String[]
		NumberFormat		nf		= NumberFormat.getInstance(Locale.US);
		nf.setGroupingUsed(false);

		try
		{
			// open the file
			BufferedReader r = new BufferedReader(new FileReader(path));
			while ((line = r.readLine()) != null)		// each element of the array list: "line" is actually a string[4]
				aL.add(line.trim().split("\\s+"));		// \\s+ - matches sequence of one or more whitespace characters.

			r.close();
		}
		catch (Exception e)
		{
			throw new Exception("The file\n" + path + "\nhas an unsupported file format!");
		}

		// go over all elements of aL
		if (col == 0)
		{
			matrix = new double[aL.size()][5];

			for	(j = 0; j < aL.size(); j++)
			{
				// aL.get(j) corresponds to one element(string[4]) of aL, that is: one entry of data containing x,y,ux,uy,L
				// check if we have the fifth column which correspond to the vector magnitude in our data.
				if (aL.get(j).length > 4)
				{
					for (i = 0; i < 5; i++)
					{
//						matrix[j][i] = Double.parseDouble(aL.get(j)[i]);
						matrix[j][i] = nf.parse(aL.get(j)[i]).doubleValue();
					}
				}
				else if (aL.get(j).length == 4)
				{
					for (i = 0; i < 4; i++)
					{
//						matrix[j][i] = Double.parseDouble(aL.get(j)[i]);
						matrix[j][i] = nf.parse(aL.get(j)[i]).doubleValue();
					}
					matrix[j][4] = Math.sqrt(matrix[j][2] * matrix[j][2] + matrix[j][3] * matrix[j][3]);
				}
				else
					throw new Exception("The file\n" + path + "\nmust have at least 4 columns (x, y, dx, dy) separated by spaces or tabs!");
			}
		}
		else
		{
			matrix = new double[aL.size()][col];

			for(j = 0; j < aL.size(); j++)
			{
				// aL.get(j) corresponds to one element(string[4]) of aL, that is: one entry of data containing x,y,ux,uy,L
				// check if we have the fifth column which correspond to the vector magnitude in our data.
				if (aL.get(j).length < col)
					throw new Exception("The file\n" + path + "\nmust have at least " + col + " columns separated by spaces or tabs!");
				else
					for (i = 0; i < col; i++)
//						matrix[j][i] = Double.parseDouble(aL.get(j)[i]);
						matrix[j][i] = nf.parse(aL.get(j)[i]).doubleValue();
			}
		}

		return matrix;
	}

   /**
	 *extract one element from a 2D array [nData points][nElements]
	 *into another 2D data array with this specified element indexed by
	 * [nData points in x][nData points in y]
	 */
	protected static double[][] get2DElement(double[][] _piv, int nx, int ny, int nEle)
	{
		int				i, j;
		boolean 		fx;
		double	[][]	new2D = new double[nx][ny];

		if (_piv[0][0] == _piv[1][0]) 			// check the _piv array is fixing x first or y first
			fx = true;							// x coordinates is fixed first
		else
			fx = false;

		for (j = 0; j < ny; j++)
			for (i = 0; i < nx; i++)
				if (fx)
					new2D[i][j] = _piv[ny * i + j][nEle];
				else
					new2D[i][j] = _piv[nx * j + i][nEle];

		return new2D;
	}
}